package cn.kk.spring_simple_operation.utils;

import cn.kk.spring_simple_operation.utils.Excel.ColumnType;
import cn.kk.spring_simple_operation.utils.Excel.Type;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.util.IOUtils;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFClientAnchor;
import org.apache.poi.xssf.usermodel.XSSFDataValidation;
import org.springframework.util.StringUtils;

import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.text.DecimalFormat;
import java.time.Instant;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Excel相关处理
 *
 * @author Devin
 */
public class ExcelUtil<T> {
	/**
	 * Excel sheet最大行数，默认65536
	 */
	public static final int    sheetSize = 65536;
	/**
	 * 工作表名称
	 */
	private             String sheetName;

	/**
	 * 导出类型（EXPORT:导出数据；IMPORT：导入模板）
	 */
	private Type type;

	/**
	 * 工作薄对象
	 */
	private Workbook wb;

	/**
	 * 工作表对象
	 */
	private Sheet sheet;

	/**
	 * 样式列表
	 */
	private Map<String, CellStyle> styles;

	/**
	 * 导入导出数据列表
	 */
	private List<T> list;

	/**
	 * 注解列表
	 */
	private List<Object[]> fields;

	/**
	 * 最大高度
	 */
	private short maxHeight;

	/**
	 * 统计列表
	 */
	private final Map<Integer, Double> statistics = new HashMap<Integer, Double>();

	/**
	 * 数字格式
	 */
	private static final DecimalFormat DOUBLE_FORMAT = new DecimalFormat("######0.00");

	/**
	 * 实体对象
	 */
	public Class<T> clazz;

	public ExcelUtil(Class<T> clazz) {
		this.clazz = clazz;
	}

	public void init(List<T> list, String sheetName, Type type) {
		if (list == null) {
			list = new ArrayList<T>();
		}
		this.list = list;
		this.sheetName = sheetName;
		this.type = type;
		createExcelField();
		createWorkbook();
	}

	/**
	 * 对list数据源将其里面的数据导入到excel表单
	 *
	 * @param response  返回数据
	 * @param list      导出数据集合
	 * @param sheetName 工作表的名称
	 * @return 结果
	 * @throws IOException
	 */
	public void exportExcel(HttpServletResponse response, List<T> list, String sheetName) throws IOException {
		response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
		response.setCharacterEncoding("utf-8");
		this.init(list, sheetName, Type.EXPORT);
		exportExcel(response.getOutputStream());
	}

	/**
	 * @param list
	 * @param sheetName
	 * @return
	 */
	@SuppressWarnings("all")
	public InputStream exporInputStream(List<T> list) {
		ByteArrayOutputStream out = null;
		try {
			this.init(list, DateUtils.getCurrentTimeSecond().toString(), Type.EXPORT);
			writeSheet();
			out = new ByteArrayOutputStream();
			wb.write(out);
			return new ByteArrayInputStream(out.toByteArray());
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			IOUtils.closeQuietly(wb);
			IOUtils.closeQuietly(out);
		}
		return null;
	}

	/**
	 * 对list数据源将其里面的数据导入到excel表单
	 *
	 * @return 结果
	 */
	public void exportExcel(OutputStream out) {
		try {
			writeSheet();
			wb.write(out);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			IOUtils.closeQuietly(wb);
			IOUtils.closeQuietly(out);
		}
	}

	/**
	 * 创建写入数据到Sheet
	 */
	public void writeSheet() {
		// 取出一共有多少个sheet.
		double sheetNo = Math.ceil(list.size() / sheetSize);
		for (int index = 0; index <= sheetNo; index++) {
			createSheet(sheetNo, index);

			// 产生一行
			Row row = sheet.createRow(0);
			int column = 0;
			// 写入各个字段的列头名称
			for (Object[] os : fields) {
				Excel excel = (Excel) os[1];
				this.createCell(excel, row, column++);
			}
			if (Type.EXPORT.equals(type)) {
				fillExcelData(index, row);
				addStatisticsRow();
			}
		}
	}

	/**
	 * 填充excel数据
	 *
	 * @param index 序号
	 * @param row   单元格行
	 */
	public void fillExcelData(int index, Row row) {
		int startNo = index * sheetSize;
		int endNo = Math.min(startNo + sheetSize, list.size());
		for (int i = startNo; i < endNo; i++) {
			row = sheet.createRow(i + 1 - startNo);
			// 得到导出对象.
			T vo = list.get(i);
			int column = 0;
			for (Object[] os : fields) {
				Field field = (Field) os[0];
				Excel excel = (Excel) os[1];
				// 设置实体类私有属性可访问
				field.setAccessible(true);
				this.addCell(excel, row, vo, field, column++);
			}
		}
	}

	/**
	 * 创建表格样式
	 *
	 * @param wb 工作薄对象
	 * @return 样式列表
	 */
	private Map<String, CellStyle> createStyles(Workbook wb) {
		// 写入各条记录,每条记录对应excel表中的一行
		Map<String, CellStyle> styles = new HashMap<String, CellStyle>();
		CellStyle style = wb.createCellStyle();
		style.setAlignment(HorizontalAlignment.CENTER);
		style.setVerticalAlignment(VerticalAlignment.CENTER);
		style.setBorderRight(BorderStyle.THIN);
		style.setRightBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
		style.setBorderLeft(BorderStyle.THIN);
		style.setLeftBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
		style.setBorderTop(BorderStyle.THIN);
		style.setTopBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
		style.setBorderBottom(BorderStyle.THIN);
		style.setBottomBorderColor(IndexedColors.GREY_50_PERCENT.getIndex());
		Font dataFont = wb.createFont();
		dataFont.setFontName("Arial");
		dataFont.setFontHeightInPoints((short) 10);
		style.setFont(dataFont);
		styles.put("data", style);

		style = wb.createCellStyle();
		style.cloneStyleFrom(styles.get("data"));
		style.setAlignment(HorizontalAlignment.CENTER);
		style.setVerticalAlignment(VerticalAlignment.CENTER);
		style.setFillForegroundColor(IndexedColors.GREY_50_PERCENT.getIndex());
		style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
		Font headerFont = wb.createFont();
		headerFont.setFontName("Arial");
		headerFont.setFontHeightInPoints((short) 10);
		headerFont.setBold(true);
		headerFont.setColor(IndexedColors.WHITE.getIndex());
		style.setFont(headerFont);
		styles.put("header", style);

		style = wb.createCellStyle();
		style.setAlignment(HorizontalAlignment.CENTER);
		style.setVerticalAlignment(VerticalAlignment.CENTER);
		Font totalFont = wb.createFont();
		totalFont.setFontName("Arial");
		totalFont.setFontHeightInPoints((short) 10);
		style.setFont(totalFont);
		styles.put("total", style);

		style = wb.createCellStyle();
		style.cloneStyleFrom(styles.get("data"));
		style.setAlignment(HorizontalAlignment.LEFT);
		styles.put("data1", style);

		style = wb.createCellStyle();
		style.cloneStyleFrom(styles.get("data"));
		style.setAlignment(HorizontalAlignment.CENTER);
		styles.put("data2", style);

		style = wb.createCellStyle();
		style.cloneStyleFrom(styles.get("data"));
		style.setAlignment(HorizontalAlignment.RIGHT);
		styles.put("data3", style);

		return styles;
	}

	/**
	 * 创建单元格
	 */
	public Cell createCell(Excel attr, Row row, int column) {
		// 创建列
		Cell cell = row.createCell(column);
		// 写入列信息
		cell.setCellValue(attr.name());
		setDataValidation(attr, row, column);
		cell.setCellStyle(styles.get("header"));
		return cell;
	}

	/**
	 * 设置单元格信息
	 *
	 * @param value 单元格值
	 * @param attr  注解相关
	 * @param cell  单元格信息
	 */
	public void setCellVo(Object value, Excel attr, Cell cell) {
		if (ColumnType.STRING == attr.cellType()) {
			cell.setCellValue(value == null ? attr.defaultValue() : value + attr.suffix());
		}
		else if (ColumnType.NUMERIC == attr.cellType()) {
			if (value != null) {
				cell.setCellValue(org.apache.commons.lang3.StringUtils.contains(TypeUtils.toStr(value), ".") ? TypeUtils.toDouble(value) : TypeUtils.toInt(value));
			}
		}
		else if (ColumnType.IMAGE == attr.cellType()) {
			ClientAnchor anchor = new XSSFClientAnchor(0, 0, 0, 0, (short) cell.getColumnIndex(), cell.getRow().getRowNum(), (short) (cell.getColumnIndex() + 1), cell.getRow().getRowNum() + 1);
			String imagePath = TypeUtils.toStr(value);
			if (!StringUtils.isEmpty(imagePath)) {
				byte[] data = ImageUtils.getImage(imagePath);
				getDrawingPatriarch(cell.getSheet()).createPicture(anchor,
						cell.getSheet().getWorkbook().addPicture(data, getImageType(data)));
			}
		}
	}

	/**
	 * 获取画布
	 */
	public static Drawing<?> getDrawingPatriarch(Sheet sheet) {
		if (sheet.getDrawingPatriarch() == null) {
			sheet.createDrawingPatriarch();
		}
		return sheet.getDrawingPatriarch();
	}

	/**
	 * 获取图片类型,设置图片插入类型
	 */
	public int getImageType(byte[] value) {
		String type = getFileExtendName(value);
		if ("JPG".equalsIgnoreCase(type)) {
			return Workbook.PICTURE_TYPE_JPEG;
		}
		else if ("PNG".equalsIgnoreCase(type)) {
			return Workbook.PICTURE_TYPE_PNG;
		}
		return Workbook.PICTURE_TYPE_JPEG;
	}

	/**
	 * 创建表格样式
	 */
	public void setDataValidation(Excel attr, Row row, int column) {
		if (attr.name().indexOf("注：") >= 0) {
			sheet.setColumnWidth(column, 6000);
		}
		else {
			// 设置列宽
			sheet.setColumnWidth(column, (int) ((attr.width() + 0.72) * 256));
		}
		// 如果设置了提示信息则鼠标放上去提示.
		if (!StringUtils.isEmpty(attr.prompt())) {
			// 这里默认设了2-101列提示.
			setXSSFPrompt(sheet, "", attr.prompt(), 1, 100, column, column);
		}
		// 如果设置了combo属性则本列只能选择不能输入
		if (attr.combo().length > 0) {
			// 这里默认设了2-101列只能选择不能输入.
			setXSSFValidation(sheet, attr.combo(), 1, 100, column, column);
		}
	}

	/**
	 * 添加单元格
	 */
	public Cell addCell(Excel attr, Row row, T vo, Field field, int column) {
		Cell cell = null;
		try {
			// 设置行高
			row.setHeight(maxHeight);
			// 根据Excel中设置情况决定是否导出,有些情况需要保持为空,希望用户填写这一列.
			if (attr.isExport()) {
				// 创建cell
				cell = row.createCell(column);
				int align = attr.align().value();
				String keyStyle = "data" + (align >= 1 && align <= 3 ? align : "");
				cell.setCellStyle(styles.get(keyStyle));

				// 用于读取对象中的属性
				Object value = getTargetValue(vo, field, attr);
				String dateFormat = attr.dateFormat();
				String readConverterExp = attr.readConverterExp();
				String separator = attr.separator();
				if (!StringUtils.isEmpty(dateFormat) && value != null && "0".equals(value.toString())) {
					cell.setCellValue("");
				}
				else if (!StringUtils.isEmpty(dateFormat) && value != null && value.toString().length() == 10) {
					//临时方案 为了把int 时间戳转成时间类型
					Integer valueInteger = Integer.valueOf(value.toString());
					Long timeLong = Long.valueOf(valueInteger * 1000L);
					Date date = new Date(timeLong);
					cell.setCellValue(DateUtils.parseDateToStr(dateFormat, date));
				}
				else if (!StringUtils.isEmpty(dateFormat) && value != null) {
					cell.setCellValue(DateUtils.parseDateToStr(dateFormat, (Date) value));
				}
				else if (!StringUtils.isEmpty(readConverterExp) && value != null) {
					cell.setCellValue(convertByExp(TypeUtils.toStr(value), readConverterExp, separator));
				}
				else if (value instanceof BigDecimal && -1 != attr.scale()) {
					cell.setCellValue((((BigDecimal) value).setScale(attr.scale(), attr.roundingMode())).toString());
				}
				else if (value instanceof Number) {
					cell.setCellValue(org.apache.commons.lang3.StringUtils.contains(TypeUtils.toStr(value), ".") ? TypeUtils.toDouble(value) : TypeUtils.toInt(value));
					// 给满足条件的文本设置颜色
					int num = ((Number) value).intValue();
					if (attr.color() >= 0 && num >= attr.colorMin() && num <= attr.colorMax()) {
						String keyColor = "data" + (align >= 1 && align <= 3 ? align : "") + "color" + attr.color();
						CellStyle cellStyle = styles.get(keyColor);
						if (Objects.isNull(cellStyle)) {
							cellStyle = this.wb.createCellStyle();
							cellStyle.cloneStyleFrom(styles.get(keyStyle));

							Font font = this.wb.createFont();
							font.setColor(attr.color());
							cellStyle.setFont(font);
							styles.put(keyColor, cellStyle);
						}
						cell.setCellStyle(cellStyle);
					}
				}
				else {
					// 设置列类型
					setCellVo(value, attr, cell);
				}
				addStatisticsData(column, TypeUtils.toStr(value), attr);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return cell;
	}

	/**
	 * 设置 POI XSSFSheet 单元格提示
	 *
	 * @param sheet         表单
	 * @param promptTitle   提示标题
	 * @param promptContent 提示内容
	 * @param firstRow      开始行
	 * @param endRow        结束行
	 * @param firstCol      开始列
	 * @param endCol        结束列
	 */
	public void setXSSFPrompt(Sheet sheet, String promptTitle, String promptContent, int firstRow, int endRow,
	                          int firstCol, int endCol) {
		DataValidationHelper helper = sheet.getDataValidationHelper();
		DataValidationConstraint constraint = helper.createCustomConstraint("DD1");
		CellRangeAddressList regions = new CellRangeAddressList(firstRow, endRow, firstCol, endCol);
		DataValidation dataValidation = helper.createValidation(constraint, regions);
		dataValidation.createPromptBox(promptTitle, promptContent);
		dataValidation.setShowPromptBox(true);
		sheet.addValidationData(dataValidation);
	}

	/**
	 * 设置某些列的值只能输入预制的数据,显示下拉框.
	 *
	 * @param sheet    要设置的sheet.
	 * @param textlist 下拉框显示的内容
	 * @param firstRow 开始行
	 * @param endRow   结束行
	 * @param firstCol 开始列
	 * @param endCol   结束列
	 * @return 设置好的sheet.
	 */
	public void setXSSFValidation(Sheet sheet, String[] textlist, int firstRow, int endRow, int firstCol, int endCol) {
		DataValidationHelper helper = sheet.getDataValidationHelper();
		// 加载下拉列表内容
		DataValidationConstraint constraint = helper.createExplicitListConstraint(textlist);
		// 设置数据有效性加载在哪个单元格上,四个参数分别是：起始行、终止行、起始列、终止列
		CellRangeAddressList regions = new CellRangeAddressList(firstRow, endRow, firstCol, endCol);
		// 数据有效性对象
		DataValidation dataValidation = helper.createValidation(constraint, regions);
		// 处理Excel兼容性问题
		if (dataValidation instanceof XSSFDataValidation) {
			dataValidation.setSuppressDropDownArrow(true);
			dataValidation.setShowErrorBox(true);
		}
		else {
			dataValidation.setSuppressDropDownArrow(false);
		}

		sheet.addValidationData(dataValidation);
	}

	/**
	 * 解析导出值 0=男,1=女,2=未知
	 *
	 * @param propertyValue 参数值
	 * @param converterExp  翻译注解
	 * @param separator     分隔符
	 * @return 解析后值
	 */
	public static String convertByExp(String propertyValue, String converterExp, String separator) {
		StringBuilder propertyString = new StringBuilder();
		String[] convertSource = converterExp.split(",");
		for (String item : convertSource) {
			String[] itemArray = item.split("=");

			if (org.apache.commons.lang3.StringUtils.containsAny(separator, propertyValue)) {
				for (String value : propertyValue.split(separator)) {
					if (itemArray[0].equals(value)) {
						propertyString.append(itemArray[1] + separator);
						break;
					}
				}
			}
			else {
				if (itemArray[0].equals(propertyValue)) {
					return itemArray[1];
				}
			}
		}
		return org.apache.commons.lang3.StringUtils.stripEnd(propertyString.toString(), separator);
	}

	/**
	 * 合计统计信息
	 */
	private void addStatisticsData(Integer index, String text, Excel entity) {
		if (entity != null && entity.isStatistics()) {
			Double temp = 0D;
			if (!statistics.containsKey(index)) {
				statistics.put(index, temp);
			}
			try {
				temp = Double.valueOf(text);
			} catch (NumberFormatException e) {
			}
			statistics.put(index, statistics.get(index) + temp);
		}
	}

	/**
	 * 创建统计行
	 */
	public void addStatisticsRow() {
		if (statistics.size() > 0) {
			Cell cell = null;
			Row row = sheet.createRow(sheet.getLastRowNum() + 1);
			Set<Integer> keys = statistics.keySet();
			cell = row.createCell(0);
			cell.setCellStyle(styles.get("total"));
			cell.setCellValue("合计");

			for (Integer key : keys) {
				cell = row.createCell(key);
				cell.setCellStyle(styles.get("total"));
				cell.setCellValue(DOUBLE_FORMAT.format(statistics.get(key)));
			}
			statistics.clear();
		}
	}

	/**
	 * 获取bean中的属性值
	 *
	 * @param vo    实体对象
	 * @param field 字段
	 * @param excel 注解
	 * @return 最终的属性值
	 * @throws Exception
	 */
	private Object getTargetValue(T vo, Field field, Excel excel) throws Exception {
		Object o = field.get(vo);
		if (!StringUtils.isEmpty(excel.targetAttr())) {
			String target = excel.targetAttr();
			if (target.indexOf(".") > -1) {
				String[] targets = target.split("[.]");
				for (String name : targets) {
					o = getValue(o, name);
				}
			}
			else {
				o = getValue(o, target);
			}
		}
		return o;
	}

	/**
	 * 以类的属性的get方法方法形式获取值
	 *
	 * @param o
	 * @param name
	 * @return value
	 * @throws Exception
	 */
	private Object getValue(Object o, String name) throws Exception {
		if (o != null && !StringUtils.isEmpty(name)) {
			Class<?> clazz = o.getClass();
			Field field = clazz.getDeclaredField(name);
			field.setAccessible(true);
			o = field.get(o);
		}
		return o;
	}

	/**
	 * 得到所有定义字段
	 */
	private void createExcelField() {
		this.fields = new ArrayList<Object[]>();
		List<Field> tempFields = new ArrayList<>();
		tempFields.addAll(Arrays.asList(clazz.getSuperclass().getDeclaredFields()));
		tempFields.addAll(Arrays.asList(clazz.getDeclaredFields()));
		for (Field field : tempFields) {
			// 单注解
			if (field.isAnnotationPresent(Excel.class)) {
				putToField(field, field.getAnnotation(Excel.class));
			}

			// 多注解
			if (field.isAnnotationPresent(Excels.class)) {
				Excels attrs = field.getAnnotation(Excels.class);
				Excel[] excels = attrs.value();
				for (Excel excel : excels) {
					putToField(field, excel);
				}
			}
		}
		this.fields = this.fields.stream().sorted(Comparator.comparing(objects -> ((Excel) objects[1]).sort())).collect(Collectors.toList());
		this.maxHeight = getRowHeight();
	}

	/**
	 * 根据注解获取最大行高
	 */
	public short getRowHeight() {
		double maxHeight = 0;
		for (Object[] os : this.fields) {
			Excel excel = (Excel) os[1];
			maxHeight = maxHeight > excel.height() ? maxHeight : excel.height();
		}
		return (short) (maxHeight * 20);
	}

	/**
	 * 放到字段集合中
	 */
	private void putToField(Field field, Excel attr) {
		if (attr != null && (attr.type() == Type.ALL || attr.type() == type)) {
			this.fields.add(new Object[]{field, attr});
		}
	}

	/**
	 * 创建一个工作簿
	 */
	public void createWorkbook() {
		this.wb = new SXSSFWorkbook(500);
	}

	/**
	 * 创建工作表
	 *
	 * @param sheetNo sheet数量
	 * @param index   序号
	 */
	public void createSheet(double sheetNo, int index) {
		this.sheet = wb.createSheet();
		this.styles = createStyles(wb);
		// 设置工作表的名称.
		if (sheetNo == 0) {
			wb.setSheetName(index, sheetName);
		}
		else {
			wb.setSheetName(index, sheetName + index);
		}
	}

	/**
	 * 获取文件类型
	 *
	 * @param photoByte 文件字节码
	 * @return 后缀（不含".")
	 */
	public static String getFileExtendName(byte[] photoByte) {
		String strFileExtendName = "JPG";
		if ((photoByte[0] == 71) && (photoByte[1] == 73) && (photoByte[2] == 70) && (photoByte[3] == 56)
				&& ((photoByte[4] == 55) || (photoByte[4] == 57)) && (photoByte[5] == 97)) {
			strFileExtendName = "GIF";
		}
		else if ((photoByte[6] == 74) && (photoByte[7] == 70) && (photoByte[8] == 73) && (photoByte[9] == 70)) {
			strFileExtendName = "JPG";
		}
		else if ((photoByte[0] == 66) && (photoByte[1] == 77)) {
			strFileExtendName = "BMP";
		}
		else if ((photoByte[1] == 80) && (photoByte[2] == 78) && (photoByte[3] == 71)) {
			strFileExtendName = "PNG";
		}
		return strFileExtendName;
	}

	/**
	 * 对list数据源将其里面的数据导入到excel表单
	 *
	 * @param sheetName 工作表的名称
	 * @return 结果
	 */
	public void importTemplateExcel(HttpServletResponse response, String sheetName) throws IOException {
		response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
		response.setCharacterEncoding("utf-8");
		this.init(null, sheetName, Type.IMPORT);
		exportExcel(response.getOutputStream());
	}

	/**
	 * 对excel表单默认第一个索引名转换成list
	 *
	 * @param is 输入流
	 * @return 转换后集合
	 */
	public List<T> importExcel(InputStream is) throws Exception {
		return importExcel("", is);
	}

	/**
	 * 对excel表单指定表格索引名转换成list
	 *
	 * @param sheetName 表格索引名
	 * @param is        输入流
	 * @return 转换后集合
	 */
	public List<T> importExcel(String sheetName, InputStream is) throws Exception {
		this.type = Type.IMPORT;
		this.wb = WorkbookFactory.create(is);
		List<T> list = new ArrayList<T>();
		Sheet sheet = null;
		if (!StringUtils.isEmpty(sheetName)) {
			// 如果指定sheet名,则取指定sheet中的内容.
			sheet = wb.getSheet(sheetName);
		}
		else {
			// 如果传入的sheet名不存在则默认指向第1个sheet.
			sheet = wb.getSheetAt(0);
		}

		if (sheet == null) {
			throw new IOException("文件sheet不存在");
		}

		// 获取最后一个非空行的行下标，比如总行数为n，则返回的为n-1
		int rows = sheet.getLastRowNum();

		if (rows > 0) {
			// 定义一个map用于存放excel列的序号和field.
			Map<String, Integer> cellMap = new HashMap<String, Integer>();
			// 获取表头
			Row heard = sheet.getRow(0);
			for (int i = 0; i < heard.getPhysicalNumberOfCells(); i++) {
				Cell cell = heard.getCell(i);
				if (cell != null) {
					String value = this.getCellValue(heard, i).toString();
					cellMap.put(value, i);
				}
				else {
					cellMap.put(null, i);
				}
			}
			// 有数据时才处理 得到类的所有field.
			Field[] allFields = clazz.getDeclaredFields();
			// 定义一个map用于存放列的序号和field.
			Map<Integer, Field> fieldsMap = new HashMap<Integer, Field>();
			for (int col = 0; col < allFields.length; col++) {
				Field field = allFields[col];
				Excel attr = field.getAnnotation(Excel.class);
				if (attr != null && (attr.type() == Type.ALL || attr.type() == type)) {
					// 设置类的私有字段属性可访问.
					field.setAccessible(true);
					Integer column = cellMap.get(attr.name());
					if (column != null) {
						fieldsMap.put(column, field);
					}
				}
			}
			for (int i = 1; i <= rows; i++) {
				// 从第2行开始取数据,默认第一行是表头.
				Row row = sheet.getRow(i);
				// 判断当前行是否是空行
				if (isRowEmpty(row)) {
					continue;
				}
				T entity = null;
				for (Map.Entry<Integer, Field> entry : fieldsMap.entrySet()) {
					Object val = this.getCellValue(row, entry.getKey());

					// 如果不存在实例则新建.
					entity = (entity == null ? clazz.newInstance() : entity);
					// 从map中得到对应列的field.
					Field field = fieldsMap.get(entry.getKey());
					// 取得类型,并根据对象类型设置值.
					Class<?> fieldType = field.getType();
					if (String.class == fieldType) {
						String s = TypeUtils.toStr(val);
						if (org.apache.commons.lang3.StringUtils.endsWith(s, ".0")) {
							val = org.apache.commons.lang3.StringUtils.substringBefore(s, ".0");
						}
						else {
							String dateFormat = field.getAnnotation(Excel.class).dateFormat();
							if (!StringUtils.isEmpty(dateFormat)) {
								val = DateUtils.parseDateToStr(dateFormat, (Date) val);
							}
							else {
								val = TypeUtils.toStr(val);
							}
						}
					}
					else if ((Integer.TYPE == fieldType || Integer.class == fieldType) &&
							org.apache.commons.lang3.StringUtils.isNumeric(TypeUtils.toStr(val))) {
						val = TypeUtils.toInt(val);
					}
					else if (Long.TYPE == fieldType || Long.class == fieldType) {
						val = TypeUtils.toLong(val);
					}
					else if (Double.TYPE == fieldType || Double.class == fieldType) {
						val = TypeUtils.toDouble(val);
					}
					else if (Float.TYPE == fieldType || Float.class == fieldType) {
						val = TypeUtils.toFloat(val);
					}
					else if (BigDecimal.class == fieldType) {
						val = TypeUtils.toBigDecimal(val);
					}
					else if (Date.class == fieldType) {
						if (val instanceof String) {
							val = DateUtils.parseDate(val);
						}
						else if (val instanceof Double) {
							val = DateUtil.getJavaDate((Double) val);
						}
					}
					else if (Boolean.TYPE == fieldType || Boolean.class == fieldType) {
						val = TypeUtils.toBool(val, false);
					}
					if (fieldType != null) {
						Excel attr = field.getAnnotation(Excel.class);
						String propertyName = field.getName();
						if (!StringUtils.isEmpty(attr.targetAttr())) {
							propertyName = field.getName() + "." + attr.targetAttr();
						}
						else if (!StringUtils.isEmpty(attr.readConverterExp())) {
							val = reverseByExp(TypeUtils.toStr(val), attr.readConverterExp(), attr.separator());
						}
						ReflectUtils.invokeSetter(entity, propertyName, val);
					}
				}
				list.add(entity);
			}
		}
		return list;
	}

	/**
	 * 获取单元格值
	 *
	 * @param row    获取的行
	 * @param column 获取单元格列号
	 * @return 单元格值
	 */
	public Object getCellValue(Row row, int column) {
		if (row == null) {
			return row;
		}
		Object val = "";
		try {
			Cell cell = row.getCell(column);
			if (cell != null) {
				if (cell.getCellType() == CellType.NUMERIC || cell.getCellType() == CellType.FORMULA) {
					val = cell.getNumericCellValue();
					if (DateUtil.isCellDateFormatted(cell)) {
						val = DateUtil.getJavaDate((Double) val); // POI Excel 日期格式转换
					}
					else {
						if ((Double) val % 1 != 0) {
							val = new BigDecimal(val.toString());
						}
						else {
							val = new DecimalFormat("0").format(val);
						}
					}
				}
				else if (cell.getCellType() == CellType.STRING) {
					val = cell.getStringCellValue();
				}
				else if (cell.getCellType() == CellType.BOOLEAN) {
					val = cell.getBooleanCellValue();
				}
				else if (cell.getCellType() == CellType.ERROR) {
					val = cell.getErrorCellValue();
				}

			}
		} catch (Exception e) {
			return val;
		}
		return val;
	}

	/**
	 * 判断是否是空行
	 *
	 * @param row 判断的行
	 * @return
	 */
	private boolean isRowEmpty(Row row) {
		if (row == null) {
			return true;
		}
		for (int i = row.getFirstCellNum(); i < row.getLastCellNum(); i++) {
			Cell cell = row.getCell(i);
			if (cell != null && cell.getCellType() != CellType.BLANK) {
				return false;
			}
		}
		return true;
	}

	/**
	 * 反向解析值 男=0,女=1,未知=2
	 *
	 * @param propertyValue 参数值
	 * @param converterExp  翻译注解
	 * @param separator     分隔符
	 * @return 解析后值
	 */
	public static String reverseByExp(String propertyValue, String converterExp, String separator) {
		StringBuilder propertyString = new StringBuilder();
		String[] convertSource = converterExp.split(",");
		for (String item : convertSource) {
			String[] itemArray = item.split("=");
			if (org.apache.commons.lang3.StringUtils.containsAny(separator, propertyValue)) {
				for (String value : propertyValue.split(separator)) {
					if (itemArray[1].equals(value)) {
						propertyString.append(itemArray[0] + separator);
						break;
					}
				}
			}
			else {
				if (itemArray[1].equals(propertyValue)) {
					return itemArray[0];
				}
			}
		}
		return org.apache.commons.lang3.StringUtils.stripEnd(propertyString.toString(), separator);
	}

	/**
	 * 适合 excel表单 多个窗口的
	 *
	 * @param is 输入流
	 * @return 转换后集合
	 */
	public Map<String, List<T>> importManySheetExcel(InputStream is) throws Exception {
		return importManySheetExcel("", is);
	}

	/**
	 * 对excel表单指定表格索引名转换成list
	 *
	 * @param sheetName 表格索引名
	 * @param is        输入流
	 * @return 转换后集合
	 */
	public Map<String, List<T>> importManySheetExcel(String sheetName, InputStream is) throws Exception {
		//返回的数据
		Map<String, List<T>> map = new HashMap<>();

		this.type = Type.IMPORT;
		this.wb = WorkbookFactory.create(is);

		//sheet窗口数量
		Integer sheetNum = 0;
		if (!StringUtils.isEmpty(sheetName)) {
			sheetNum = 1;
		}
		else {
			sheetNum = wb.getNumberOfSheets();
		}

		for (int k = 0; k < sheetNum; k++) {
			//sheet 窗口名称
			String sheetKey = wb.getSheetName(k);

			List<T> list = new ArrayList<T>();
			Sheet sheet = null;
			if (!StringUtils.isEmpty(sheetName)) {
				// 如果指定sheet名,则取指定sheet中的内容.
				sheet = wb.getSheet(sheetName);
			}
			else {
				// 如果传入的sheet名不存在则默认指向第1个sheet.
				sheet = wb.getSheetAt(k);
			}

			if (sheet == null) {
				throw new IOException("文件sheet不存在");
			}

			// 获取最后一个非空行的行下标，比如总行数为n，则返回的为n-1
			int rows = sheet.getLastRowNum();

			if (rows > 0) {
				// 定义一个map用于存放excel列的序号和field.
				Map<String, Integer> cellMap = new HashMap<String, Integer>();
				// 获取表头
				Row heard = sheet.getRow(0);
				for (int i = 0; i < heard.getPhysicalNumberOfCells(); i++) {
					Cell cell = heard.getCell(i);
					if (cell != null) {
						String value = this.getCellValue(heard, i).toString();
						cellMap.put(value, i);
					}
					else {
						cellMap.put(null, i);
					}
				}
				// 有数据时才处理 得到类的所有field.
				Field[] allFields = clazz.getDeclaredFields();
				// 定义一个map用于存放列的序号和field.
				Map<Integer, Field> fieldsMap = new HashMap<Integer, Field>();
				for (int col = 0; col < allFields.length; col++) {
					Field field = allFields[col];
					Excel attr = field.getAnnotation(Excel.class);
					if (attr != null && (attr.type() == Type.ALL || attr.type() == type)) {
						// 设置类的私有字段属性可访问.
						field.setAccessible(true);
						Integer column = cellMap.get(attr.name());
						if (column != null) {
							fieldsMap.put(column, field);
						}
					}
				}
				for (int i = 1; i <= rows; i++) {
					// 从第2行开始取数据,默认第一行是表头.
					Row row = sheet.getRow(i);
					// 判断当前行是否是空行
					if (isRowEmpty(row)) {
						continue;
					}
					T entity = null;
					for (Map.Entry<Integer, Field> entry : fieldsMap.entrySet()) {
						Object val = this.getCellValue(row, entry.getKey());

						// 如果不存在实例则新建.
						entity = (entity == null ? clazz.newInstance() : entity);
						// 从map中得到对应列的field.
						Field field = fieldsMap.get(entry.getKey());
						// 取得类型,并根据对象类型设置值.
						Class<?> fieldType = field.getType();
						if (String.class == fieldType) {
							String s = TypeUtils.toStr(val);
							if (org.apache.commons.lang3.StringUtils.endsWith(s, ".0")) {
								val = org.apache.commons.lang3.StringUtils.substringBefore(s, ".0");
							}
							else {
								String dateFormat = field.getAnnotation(Excel.class).dateFormat();
								if (!StringUtils.isEmpty(dateFormat)) {
									val = DateUtils.parseDateToStr(dateFormat, (Date) val);
								}
								else {
									val = TypeUtils.toStr(val);
								}
							}
						}
						else if ((Integer.TYPE == fieldType || Integer.class == fieldType) &&
								org.apache.commons.lang3.StringUtils.isNumeric(TypeUtils.toStr(val))) {
							val = TypeUtils.toInt(val);
						}
						else if (Long.TYPE == fieldType || Long.class == fieldType) {
							val = TypeUtils.toLong(val);
						}
						else if (Double.TYPE == fieldType || Double.class == fieldType) {
							val = TypeUtils.toDouble(val);
						}
						else if (Float.TYPE == fieldType || Float.class == fieldType) {
							val = TypeUtils.toFloat(val);
						}
						else if (BigDecimal.class == fieldType) {
							val = TypeUtils.toBigDecimal(val);
						}
						else if (Date.class == fieldType) {
							if (val instanceof String) {
								val = DateUtils.parseDate(val);
							}
							else if (val instanceof Double) {
								val = DateUtil.getJavaDate((Double) val);
							}
						}
						else if (Boolean.TYPE == fieldType || Boolean.class == fieldType) {
							val = TypeUtils.toBool(val, false);
						}
						if (fieldType != null) {
							Excel attr = field.getAnnotation(Excel.class);
							String propertyName = field.getName();
							if (!StringUtils.isEmpty(attr.targetAttr())) {
								propertyName = field.getName() + "." + attr.targetAttr();
							}
							else if (!StringUtils.isEmpty(attr.readConverterExp())) {
								val = reverseByExp(TypeUtils.toStr(val), attr.readConverterExp(), attr.separator());
							}
							ReflectUtils.invokeSetter(entity, propertyName, val);
						}
					}
					list.add(entity);
				}
			}

			map.put(sheetKey, list);
		}

		return map;
	}

	/**
	 * 导入的excel 包含有合并列
	 *
	 * @param sheetName 窗口名 可不填
	 * @param is        文件
	 * @return
	 * @throws Exception
	 */
	public List<T> importMergeColumnsExcel(String sheetName, InputStream is) throws Exception {
		List<Object[]> objArrList = new ArrayList<>();
		this.type = Type.IMPORT;
		this.wb = WorkbookFactory.create(is);
		//返回的数据
		List<T> list = new ArrayList<T>();
		//获取Sheet
		Sheet sheet = null;
		if (!StringUtils.isEmpty(sheetName)) {
			// 如果指定sheet名,则取指定sheet中的内容.
			sheet = wb.getSheet(sheetName);
		}
		else {
			// 如果传入的sheet名不存在则默认指向第1个sheet.
			sheet = wb.getSheetAt(0);
		}

		if (sheet == null) {
			throw new IOException("文件sheet不存在");
		}

		// 获取最后一个非空行的行下标，比如总行数为n，则返回的为n-1
		int rows = sheet.getLastRowNum();

		if (rows > 0) {
			// 定义一个map用于存放excel列的序号和field.
			Map<String, Integer> cellMap = new HashMap<String, Integer>();
			// 获取表头
			Row heard = sheet.getRow(0);
			for (int i = 0; i < heard.getPhysicalNumberOfCells(); i++) {
				Cell cell = heard.getCell(i);
				if (cell != null) {
					String value = this.getCellValue(heard, i).toString();
					cellMap.put(value, i);
				}
				else {
					cellMap.put(null, i);
				}
			}

			// 有数据时才处理 得到类的所有field.
			Field[] allFields = clazz.getDeclaredFields();
			// 定义一个map用于存放列的序号和field.
			Map<Integer, Field> fieldsMap = new HashMap<Integer, Field>();
			for (int col = 0; col < allFields.length; col++) {
				Field field = allFields[col];
				Excel attr = field.getAnnotation(Excel.class);
				if (attr != null && (attr.type() == Type.ALL || attr.type() == type)) {
					// 设置类的私有字段属性可访问.
					field.setAccessible(true);
					Integer column = cellMap.get(attr.name());
					if (column != null) {
						fieldsMap.put(column, field);
					}
				}
			}

			for (int i = 1; i <= rows; i++) {
				// 从第2行开始取数据,默认第一行是表头.
				Row row = sheet.getRow(i);

				T entity = null;
				List<Object> objList = new ArrayList<>();
				for (Map.Entry<Integer, Field> entry : fieldsMap.entrySet()) {
					Cell c = row.getCell(entry.getKey());
					if (c == null) {
						objList.add("");
						continue;
					}
					boolean isMerge = isMergedRegion(sheet, i, c.getColumnIndex());
					//判断是否具有合并单元格
					Object val = "";
					if (isMerge) {
						val = getMergedRegionValue(sheet, row.getRowNum(), c.getColumnIndex());
					}
					else {
						val = getCellValue(c);
					}

					// 如果不存在实例则新建.
					entity = (entity == null ? clazz.newInstance() : entity);
					// 从map中得到对应列的field.
					Field field = fieldsMap.get(entry.getKey());
					// 取得类型,并根据对象类型设置值.
					Class<?> fieldType = field.getType();
					if (String.class == fieldType) {
						String s = TypeUtils.toStr(val);
						if (org.apache.commons.lang3.StringUtils.endsWith(s, ".0")) {
							val = org.apache.commons.lang3.StringUtils.substringBefore(s, ".0");
						}
						else {
							String dateFormat = field.getAnnotation(Excel.class).dateFormat();
							if (!StringUtils.isEmpty(dateFormat)) {
								val = DateUtils.parseDateToStr(dateFormat, (Date) val);
							}
							else {
								val = TypeUtils.toStr(val);
							}
						}
					}
					else if ((Integer.TYPE == fieldType || Integer.class == fieldType) &&
							org.apache.commons.lang3.StringUtils.isNumeric(TypeUtils.toStr(val))) {
						val = TypeUtils.toInt(val);
					}
					else if (Long.TYPE == fieldType || Long.class == fieldType) {
						val = TypeUtils.toLong(val);
					}
					else if (Double.TYPE == fieldType || Double.class == fieldType) {
						val = TypeUtils.toDouble(val);
					}
					else if (Float.TYPE == fieldType || Float.class == fieldType) {
						val = TypeUtils.toFloat(val);
					}
					else if (BigDecimal.class == fieldType) {
						val = TypeUtils.toBigDecimal(val);
					}
					else if (Date.class == fieldType) {
						if (val instanceof String) {
							val = DateUtils.parseDate(val);
						}
						else if (val instanceof Double) {
							val = DateUtil.getJavaDate((Double) val);
						}
					}
					else if (Boolean.TYPE == fieldType || Boolean.class == fieldType) {
						val = TypeUtils.toBool(val, false);
					}
					if (fieldType != null) {
						Excel attr = field.getAnnotation(Excel.class);
						String propertyName = field.getName();
						if (!StringUtils.isEmpty(attr.targetAttr())) {
							propertyName = field.getName() + "." + attr.targetAttr();
						}
						else if (!StringUtils.isEmpty(attr.readConverterExp())) {
							val = reverseByExp(TypeUtils.toStr(val), attr.readConverterExp(), attr.separator());
						}
						ReflectUtils.invokeSetter(entity, propertyName, val);
					}
				}
				list.add(entity);
			}

		}

		return list;
	}

	/**
	 * 获取合并单元格的值
	 *
	 * @param sheet
	 * @param row
	 * @param column
	 * @return
	 */
	public static Object getMergedRegionValue(Sheet sheet, int row, int column) {
		int sheetMergeCount = sheet.getNumMergedRegions();

		for (int i = 0; i < sheetMergeCount; i++) {
			CellRangeAddress ca = sheet.getMergedRegion(i);
			int firstColumn = ca.getFirstColumn();
			int lastColumn = ca.getLastColumn();
			int firstRow = ca.getFirstRow();
			int lastRow = ca.getLastRow();

			if (row >= firstRow && row <= lastRow) {

				if (column >= firstColumn && column <= lastColumn) {
					Row fRow = sheet.getRow(firstRow);
					Cell fCell = fRow.getCell(firstColumn);
					return getCellValue(fCell);
				}
			}
		}

		return null;
	}

	/**
	 * 获取单元格的值
	 *
	 * @param cell
	 * @return
	 */
	public static Object getCellValue(Cell cell) {
		Object val = "";
		if (cell != null) {
			if (cell.getCellType() == CellType.NUMERIC || cell.getCellType() == CellType.FORMULA) {
				val = cell.getNumericCellValue();
				if (DateUtil.isCellDateFormatted(cell)) {
					val = DateUtil.getJavaDate((Double) val); // POI Excel 日期格式转换
				}
				else {
					if ((Double) val % 1 != 0) {
						val = new BigDecimal(val.toString());
					}
					else {
						val = new DecimalFormat("0").format(val);
					}
				}
			}
			else if (cell.getCellType() == CellType.STRING) {
				val = cell.getStringCellValue();
			}
			else if (cell.getCellType() == CellType.BOOLEAN) {
				val = cell.getBooleanCellValue();
			}
			else if (cell.getCellType() == CellType.ERROR) {
				val = cell.getErrorCellValue();
			}
		}
		return val;
	}

	/**
	 * 判断指定的单元格是否是合并单元格
	 *
	 * @param sheet
	 * @param row    行下标
	 * @param column 列下标
	 * @return
	 */
	public static boolean isMergedRegion(Sheet sheet, int row, int column) {
		int sheetMergeCount = sheet.getNumMergedRegions();
		for (int i = 0; i < sheetMergeCount; i++) {
			CellRangeAddress range = sheet.getMergedRegion(i);
			int firstColumn = range.getFirstColumn();
			int lastColumn = range.getLastColumn();
			int firstRow = range.getFirstRow();
			int lastRow = range.getLastRow();
			if (row >= firstRow && row <= lastRow) {
				if (column >= firstColumn && column <= lastColumn) {
					return true;
				}
			}
		}
		return false;
	}

	public void exportExcelFile(List<T> list, String fileName) {
		this.init(list, fileName, Type.EXPORT);
		try {
			Path tempFile = Files.createTempFile(fileName, ".xlsx");
			BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(tempFile));
			exportExcel(out);

			Path path = Files.createFile(Paths.get(fileName + "-" + Instant.now().getEpochSecond() + ".xlsx"));
			Files.copy(tempFile, path, StandardCopyOption.REPLACE_EXISTING);
			Files.deleteIfExists(tempFile);

		} catch (IOException e) {
			e.printStackTrace();
		}

	}
}