package cn.kk.spring_simple_operation.utils;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.poi.ss.usermodel.DateUtil;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.*;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 反射工具类. 提供调用getter/setter方法, 访问私有变量, 调用私有方法, 获取泛型类型Class, 被AOP过的真实类等工具函数.
 */
@SuppressWarnings("rawtypes")
public class ReflectUtils {
	private static final Map<String, PropertyDescriptor> propertyDescriptorCache = new ConcurrentHashMap<>();

	private static final String SETTER_PREFIX = "set";

	private static final String GETTER_PREFIX = "get";

	private static final String CGLIB_CLASS_SEPARATOR = "$$";


	public static <E> void invokeSetterFast(Object obj, String propertyName, E value) {
		if (obj == null) return;
		Assert.isTrue(StringUtils.isNotBlank(propertyName), "propertyName 不能为空!!");
		String clazzName = obj.getClass().getName();
		PropertyDescriptor descriptor = ReflectUtils.getPropertyDescriptor(obj, propertyName);
		Assert.notNull(descriptor, String.format("类【%s】的属性【%s】获取读方法失败!!", clazzName, propertyName));
		Class<?> propertyType = descriptor.getPropertyType();
		if (value != null) {
			Assert.isTrue(propertyType.equals(value.getClass()), "反射value对应的字段类型不匹配,请检查!!");
		}
		Method writeMethod = descriptor.getWriteMethod();
		writeMethod.setAccessible(true);
		try {
			ReflectionUtils.invokeMethod(writeMethod, obj, value);
		} catch (Exception e) {
			throw new RuntimeException("反射set方法调用失败!:" + e.getMessage());
		}
	}

	public static Object invokeGetterFast(Object obj, String propertyName) {
		if (obj == null) return null;
		Assert.isTrue(StringUtils.isNotBlank(propertyName), "propertyName 不能为空!!");
		String clazzName = obj.getClass().getName();
		PropertyDescriptor descriptor = ReflectUtils.getPropertyDescriptor(obj, propertyName);
		Assert.notNull(descriptor, String.format("类【%s】的属性【%s】获取写方法失败!!", clazzName, propertyName));
		Method readMethod = descriptor.getReadMethod();
		readMethod.setAccessible(true);
		try {
			return ReflectionUtils.invokeMethod(readMethod, obj);
		} catch (Exception e) {
			throw new RuntimeException("反射set方法调用失败!:" + e.getMessage());
		}
	}

	/**
	 * 获取对象属性描述
	 *
	 * @param obj          对象
	 * @param propertyName 属性
	 */
	private static PropertyDescriptor getPropertyDescriptor(Object obj, String propertyName) {
		String clazzName = obj.getClass().getName();
		String key = clazzName + "::" + propertyName;
		return propertyDescriptorCache.computeIfAbsent(key, s -> {
			BeanWrapperImpl wrapper = new BeanWrapperImpl(obj);
			PropertyDescriptor[] descriptors = wrapper.getPropertyDescriptors();
			for (PropertyDescriptor descriptor : descriptors) {
				String name = descriptor.getName();
				if (propertyName.equals(name)) {
					return descriptor;
				}
			}
			return null;
		});
	}

	/**
	 * 调用Getter方法.
	 * 支持多级，如：对象名.对象名.方法
	 */
	@SuppressWarnings("unchecked")
	public static <E> E invokeGetter(Object obj, String propertyName) {
		Object object = obj;
		for (String name : StringUtils.split(propertyName, ".")) {
			String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(name);
			object = invokeMethod(object, getterMethodName, new Class[]{}, new Object[]{});
		}
		return (E) object;
	}

	/**
	 * 调用Setter方法, 仅匹配方法名。
	 * 支持多级，如：对象名.对象名.方法
	 */
	public static <E> void invokeSetter(Object obj, String propertyName, E value) {
		Object object = obj;
		String[] names = StringUtils.split(propertyName, ".");
		for (int i = 0; i < names.length; i++) {
			if (i < names.length - 1) {
				String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(names[i]);
				object = invokeMethod(object, getterMethodName, new Class[]{}, new Object[]{});
			} else {
				String setterMethodName = SETTER_PREFIX + StringUtils.capitalize(names[i]);
				invokeMethodByName(object, setterMethodName, new Object[]{value});
			}
		}
	}

	/**
	 * 直接读取对象属性值, 无视private/protected修饰符, 不经过getter函数.
	 */
	@SuppressWarnings("unchecked")
	public static <E> E getFieldValue(final Object obj, final String fieldName) {
		Field field = getAccessibleField(obj, fieldName);
		if (field == null) {
			return null;
		}
		E result = null;
		try {
			result = (E) field.get(obj);
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		}
		return result;
	}

	/**
	 * 直接设置对象属性值, 无视private/protected修饰符, 不经过setter函数.
	 */
	public static <E> void setFieldValue(final Object obj, final String fieldName, final E value) {
		Field field = getAccessibleField(obj, fieldName);
		if (field == null) {
			return;
		}
		try {
			field.set(obj, value);
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 直接调用对象方法, 无视private/protected修饰符.
	 * 用于一次性调用的情况，否则应使用getAccessibleMethod()函数获得Method后反复调用.
	 * 同时匹配方法名+参数类型，
	 */
	@SuppressWarnings("unchecked")
	public static <E> E invokeMethod(final Object obj, final String methodName, final Class<?>[] parameterTypes,
	                                 final Object[] args) {
		if (obj == null || methodName == null) {
			return null;
		}
		Method method = getAccessibleMethod(obj, methodName, parameterTypes);
		if (method == null) {
			return null;
		}
		try {
			return (E) method.invoke(obj, args);
		} catch (Exception e) {
			String msg = "method: " + method + ", obj: " + obj + ", args: " + args + "";
			throw convertReflectionExceptionToUnchecked(msg, e);
		}
	}

	/**
	 * 直接调用对象方法, 无视private/protected修饰符，
	 * 用于一次性调用的情况，否则应使用getAccessibleMethodByName()函数获得Method后反复调用.
	 * 只匹配函数名，如果有多个同名函数调用第一个。
	 */
	@SuppressWarnings("unchecked")
	public static <E> E invokeMethodByName(final Object obj, final String methodName, final Object[] args) {
		Method method = getAccessibleMethodByName(obj, methodName, args.length);
		if (method == null) {
			// 如果为空不报错，直接返回空。
			return null;
		}
		try {
			// 类型转换（将参数数据类型转换为目标方法参数类型）
			Class<?>[] cs = method.getParameterTypes();
			for (int i = 0; i < cs.length; i++) {
				if (args[i] != null && !args[i].getClass().equals(cs[i])) {
					if (cs[i] == String.class) {
						args[i] = TypeUtils.toStr(args[i]);
						if (StringUtils.endsWith((String) args[i], ".0")) {
							args[i] = StringUtils.substringBefore((String) args[i], ".0");
						}
					} else if (cs[i] == Integer.class) {
						args[i] = TypeUtils.toInt(args[i]);
					} else if (cs[i] == Long.class) {
						args[i] = TypeUtils.toLong(args[i]);
					} else if (cs[i] == Double.class) {
						args[i] = TypeUtils.toDouble(args[i]);
					} else if (cs[i] == Float.class) {
						args[i] = TypeUtils.toFloat(args[i]);
					} else if (cs[i] == Date.class) {
						if (args[i] instanceof String) {
							args[i] = DateUtils.parseDate(args[i]);
						} else {
							args[i] = DateUtil.getJavaDate((Double) args[i]);
						}
					} else if (cs[i] == boolean.class || cs[i] == Boolean.class) {
						args[i] = TypeUtils.toBool(args[i]);
					}
				}
			}
			return (E) method.invoke(obj, args);
		} catch (Exception e) {
			String msg = "method: " + method + ", obj: " + obj + ", args: " + args + "";
			throw convertReflectionExceptionToUnchecked(msg, e);
		}
	}

	/**
	 * 循环向上转型, 获取对象的DeclaredField, 并强制设置为可访问.
	 * 如向上转型到Object仍无法找到, 返回null.
	 */
	public static Field getAccessibleField(final Object obj, final String fieldName) {
		// 为空不报错。直接返回 null
		if (obj == null) {
			return null;
		}
		Validate.notBlank(fieldName, "fieldName can't be blank");
		for (Class<?> superClass = obj.getClass(); superClass != Object.class; superClass = superClass.getSuperclass()) {
			try {
				Field field = superClass.getDeclaredField(fieldName);
				makeAccessible(field);
				return field;
			} catch (NoSuchFieldException e) {
				continue;
			}
		}
		return null;
	}

	/**
	 * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问.
	 * 如向上转型到Object仍无法找到, 返回null.
	 * 匹配函数名+参数类型。
	 * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
	 */
	public static Method getAccessibleMethod(final Object obj, final String methodName,
	                                         final Class<?>... parameterTypes) {
		// 为空不报错。直接返回 null
		if (obj == null) {
			return null;
		}
		Validate.notBlank(methodName, "methodName can't be blank");
		for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
			try {
				Method method = searchType.getDeclaredMethod(methodName, parameterTypes);
				makeAccessible(method);
				return method;
			} catch (NoSuchMethodException e) {
				continue;
			}
		}
		return null;
	}

	/**
	 * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问.
	 * 如向上转型到Object仍无法找到, 返回null.
	 * 只匹配函数名。
	 * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
	 */
	public static Method getAccessibleMethodByName(final Object obj, final String methodName, int argsNum) {
		// 为空不报错。直接返回 null
		if (obj == null) {
			return null;
		}
		Validate.notBlank(methodName, "methodName can't be blank");
		for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
			Method[] methods = searchType.getDeclaredMethods();
			for (Method method : methods) {
				if (method.getName().equals(methodName) && method.getParameterTypes().length == argsNum) {
					makeAccessible(method);
					return method;
				}
			}
		}
		return null;
	}

	/**
	 * 改变private/protected的方法为public，尽量不调用实际改动的语句，避免JDK的SecurityManager抱怨。
	 */
	public static void makeAccessible(Method method) {
		if ((!Modifier.isPublic(method.getModifiers()) || !Modifier.isPublic(method.getDeclaringClass().getModifiers()))
				&& !method.isAccessible()) {
			method.setAccessible(true);
		}
	}

	/**
	 * 改变private/protected的成员变量为public，尽量不调用实际改动的语句，避免JDK的SecurityManager抱怨。
	 */
	public static void makeAccessible(Field field) {
		if ((!Modifier.isPublic(field.getModifiers()) || !Modifier.isPublic(field.getDeclaringClass().getModifiers())
				|| Modifier.isFinal(field.getModifiers())) && !field.isAccessible()) {
			field.setAccessible(true);
		}
	}

	/**
	 * 通过反射, 获得Class定义中声明的泛型参数的类型, 注意泛型必须定义在父类处
	 * 如无法找到, 返回Object.class.
	 */
	@SuppressWarnings("unchecked")
	public static <T> Class<T> getClassGenricType(final Class clazz) {
		return getClassGenricType(clazz, 0);
	}

	/**
	 * 通过反射, 获得Class定义中声明的父类的泛型参数的类型.
	 * 如无法找到, 返回Object.class.
	 */
	public static Class getClassGenricType(final Class clazz, final int index) {
		Type genType = clazz.getGenericSuperclass();

		if (!(genType instanceof ParameterizedType)) {
			return Object.class;
		}

		Type[] params = ((ParameterizedType) genType).getActualTypeArguments();

		if (index >= params.length || index < 0) {
			return Object.class;
		}
		if (!(params[index] instanceof Class)) {
			return Object.class;
		}

		return (Class) params[index];
	}

	public static Class<?> getUserClass(Object instance) {
		if (instance == null) {
			throw new RuntimeException("Instance must not be null");
		}
		Class clazz = instance.getClass();
		if (clazz != null && clazz.getName().contains(CGLIB_CLASS_SEPARATOR)) {
			Class<?> superClass = clazz.getSuperclass();
			if (superClass != null && !Object.class.equals(superClass)) {
				return superClass;
			}
		}
		return clazz;

	}

	/**
	 * 将反射时的checked exception转换为unchecked exception.
	 */
	public static RuntimeException convertReflectionExceptionToUnchecked(String msg, Exception e) {
		if (e instanceof IllegalAccessException || e instanceof IllegalArgumentException
				|| e instanceof NoSuchMethodException) {
			return new IllegalArgumentException(msg, e);
		} else if (e instanceof InvocationTargetException) {
			return new RuntimeException(msg, ((InvocationTargetException) e).getTargetException());
		}
		return new RuntimeException(msg, e);
	}


	/**
	 * 判断对象属性除了id是否都为空
	 */
	public static boolean isNotNullWithId(Object object) {
		// 获取object的Class对象
		Class<?> clazz = object.getClass();
		// 获取对象的所有属性
		Field[] fields = clazz.getDeclaredFields();
		// 定义返回结果
		boolean flag = false;

		for (Field field : fields) {
			if ("id".equals(field.getName()) || "serialVersionUID".equals(field.getName())) {
				continue;
			}
			// 使非Public类型的属性可以被访问
			field.setAccessible(true);
			Object fieldValue = null;

			try {
				fieldValue = field.get(object);
				// 获取到属性类型
				Type type = field.getType();
				// 获取属性名称


			} catch (Exception e) {
				e.printStackTrace();
			}

			// 只要有一个属性值不为null 就返回false 表示对象不为null
			if (fieldValue != null) {
				flag = true;
				break;
			}
		}

		return flag;
	}
}
