import os
import time
import threading
import queue
import csv
from datetime import datetime
from typing import List, Dict, Tuple
import app_amazon_image_recognition as amazon_recog
from loguru import logger

# 测试配置
TEST_FOLDER = "temp_img"  # 图片文件夹路径
REQUESTS_PER_IMAGE = 1  # 每张图片请求次数
THREAD_COUNT = 10  # 线程数量
RESULT_CSV = "test_results.csv"  # 结果保存文件

# 线程安全的队列用于分发任务
task_queue = queue.Queue()
# 结果存储（线程安全）
result_lock = threading.Lock()
test_results = []


def worker(cookies: dict):
    """线程工作函数，处理队列中的任务"""
    thread_name = threading.current_thread().name  # 获取线程名称（如Worker-1）
    while not task_queue.empty():
        try:
            # # 从队列获取任务（图片路径+当前请求序号），非阻塞模式
            image_path, request_idx = task_queue.get(block=False)
            image_name = os.path.basename(image_path)

            start_time = time.time()
            start_datetime = datetime.now()

            # 打印开始请求的日志（关键：展示线程、图片、序号）
            logger.info(f"[{thread_name}] 开始处理 {image_name} 第 {request_idx} 次请求")

            # 执行请求 传入cookies.copy()避免多线程共享同一字典导致的修改冲突
            result_url = amazon_recog.get_amazon_search_url(image_path, cookies.copy())

            # 计算耗时
            end_time = time.time()
            duration = end_time - start_time
            #  标记请求是否成功（1=成功，0=失败）
            success = 1 if result_url else 0

            # 记录结果
            with result_lock:
                test_results.append({
                    "thread_name": thread_name,  # 记录线程名
                    "image": image_name, # 图片文件名
                    "request_index": request_idx,  # 第几次请求（1到1000）
                    "start_time": start_datetime.strftime("%Y-%m-%d %H:%M:%S.%f"), # 开始时间
                    "duration": round(duration, 4),  # 耗时 保留4位小数）
                    "success": success,  # 是否成功
                    "result_url": result_url if success else "None", # 成功时的URL，失败为None
                    "thread_id": threading.current_thread().ident  # 处理线程ID
                })

            # 每完成100次请求打印进度
            if request_idx % 100 == 0:
                logger.info(f"图片 {os.path.basename(image_path)} 完成第 {request_idx} 次请求，耗时 {duration:.2f}s")

        except queue.Empty:
            # 队列已空，退出循环
            break
        except Exception as e:
            logger.error(f"线程处理出错: {str(e)}")
        finally:
            #  队列已空，退出循环
            task_queue.task_done()


def load_test_cookies() -> dict:
    """加载测试用的cookies"""
    return {
        "i18n-prefs": "USD",
        "lc-main": "en_US",
        "session-id": "131-0347800-4175077",
        "session-id-time": "2082787201l",
        "session-token": "Jo+AthxsQrcFH8qeii+sHhoo7puFd/cpJUEsjnWXtCLhr8ycF9TQSAv9zuyoAvFjfmXZuACFNa/D+i5et63EafMMPDK/825m8TUtNtlO88KmmEsm94fiyoPL0UakTyZsUBv/CzndcKB7h0K3NkbeFws9gZSdwYRGJFVeX+pPQ9ceN0WkE+XLwCt0plIIxG3BC+VtdFJWxPKxH+R5dlnbtxPso2S5zlrOf1FTEdGNNhNZvVq25XeydshrSp7AKG6VUOicnipgfAY0Qle3Y4bw72N1IqY9i3rXVZlkrkGePamBxew+Vel7U8ccVsEIT/vtOtLHPfsTljTgltlJU0bzk0YeoJ1LwI9S",
        "skin": "noskin",
        "ubid-main": "134-9889499-1876667"
    }


def collect_image_paths(folder: str) -> List[str]:
    """收集文件夹中的所有图片路径"""
    image_extensions = ('.png', '.jpg', '.jpeg', '.gif', '.bmp')

    if not os.path.exists(folder):
        logger.error(f"图片文件夹不存在: {folder}")
        return []
    image_paths = [
        os.path.join(folder, f)
        for f in os.listdir(folder)
        if f.lower().endswith(image_extensions)
    ]
    logger.info(f"发现 {len(image_paths)} 张图片: {[os.path.basename(p) for p in image_paths]}")
    return image_paths


COLLECT_IMAGE_PATHS = collect_image_paths


def init_task_queue(image_paths: List[str]):
    """初始化任务队列 将每张图片的N次请求加入队列"""
    # for image_path in image_paths:
    #     for i in range(1, REQUESTS_PER_IMAGE + 1):
    #         task_queue.put((image_path, i))  # 每个任务是（图片路径，请求序号）
    # 轮询交替放任务（每张图片交替入队）
    for request_idx in range(1, REQUESTS_PER_IMAGE + 1):
        for image_path in image_paths:
            task_queue.put((image_path, request_idx))
    logger.info(f"任务队列初始化完成，总任务数: {task_queue.qsize()}")


def save_results_to_csv(results: List[Dict], filename: str):
    """将结果保存到CSV文件"""
    if not results:
        logger.warning("没有结果可保存")
        return

    with open(filename, 'w', newline='', encoding='utf-8') as f:
        # 以结果列表中第一个字典的键作为CSV表头
        fieldnames = results[0].keys()
        writer = csv.DictWriter(f, fieldnames=fieldnames)

        writer.writeheader()
        for row in results:
            writer.writerow(row) #  写入每行数据

    with open(RESULT_CSV, 'w', newline='', encoding='utf-8') as f:
        writer = csv.DictWriter(f, fieldnames=test_results[0].keys())
        writer.writeheader()
        writer.writerows(test_results)
    logger.info(f"结果已保存到 {RESULT_CSV}")

def print_summary(results: List[Dict]):
    """打印测试总结（成功率、耗时等关键指标）"""
    if not results:
        return

    total = len(results)
    success = sum(1 for r in test_results if r["success"])
    logger.info(f"\n===== 测试总结 =====")
    logger.info(f"总请求: {total}，成功: {success}，成功率: {success/total*100:.2f}%")

    # 按图片分组统计 （每张图片的单独表现）
    image_stats = {}
    for r in results:
        img = r["image"]
        if img not in image_stats:
            image_stats[img] = {"total": 0, "success": 0, "duration": []}
        image_stats[img]["total"] += 1
        image_stats[img]["success"] += r["success"]
        image_stats[img]["duration"].append(r["duration"])

    # 按线程统计
    thread_stats = {}
    for r in test_results:
        t = r["thread_name"]
        thread_stats[t] = thread_stats.get(t, 0) + 1
    logger.info("\n线程任务分配:")
    for t, cnt in thread_stats.items():
        logger.info(f"  {t}: {cnt} 次请求")


def main():
    # 初始化日志
    logger.add("stability_test.log", rotation="10 MB", level="INFO")

    cookies = load_test_cookies()
    image_paths = COLLECT_IMAGE_PATHS(TEST_FOLDER)
    if not image_paths:
        logger.error("没有找到图片，无法进行测试")
        return

    init_task_queue(image_paths)

    # 记录测试开始时间
    test_start_time = time.time()

    # 创建并启动线程
    threads = []
    for i in range(THREAD_COUNT):
        t = threading.Thread(target=worker, args=(cookies,), name=f"Worker-{i + 1}")
        threads.append(t)
        t.start()
        logger.info(f"启动线程: {t.name}")

    # 等待所有任务完成
    task_queue.join()

    # 等待所有线程结束
    for t in threads:
        t.join()

    # 计算总耗时
    total_test_time = time.time() - test_start_time
    logger.info(f"所有测试完成，总耗时: {total_test_time:.2f}秒")

    # 保存结果并打印总结
    save_results_to_csv(test_results, RESULT_CSV)
    print_summary(test_results)


if __name__ == "__main__":
    main()