import hashlib
import os
import time
import uuid
import random
from typing import Optional, Dict, Any
from loguru import logger
import requests
import json
from PIL import Image
from requests.exceptions import RequestException, JSONDecodeError
# 导入所有站点相关配置
from amazon_configs import (
    site_name_secret_dict,
    us_devices_list,
    uk_devices_list,
    de_devices_list,
    us_cookie_dict,
    uk_cookie_dict,
    de_cookie_dict
)
# 日志目录配置
# LOG_DIR = "log"  # 日志文件夹名称

# 关键新增：判断日志目录是否存在，不存在则创建
# if not os.path.exists(LOG_DIR):
#     os.makedirs(LOG_DIR, exist_ok=True)  # exist_ok=True 避免目录已存在时报错
# 配置日志
# logger.add("log/amazon_search_optimized.log", rotation="10 MB", level="INFO", encoding="utf-8")
# log_file = os.path.join(LOG_DIR, "amazon_search_optimized.log")
# logger.add(log_file,rotation="10 MB", level="INFO",encoding="utf-8")

# 站点配置映射（关联设备列表和Cookie） uk 和de 未测试准确性
SITE_CONFIG_MAPPER = {
    "us": {
        "devices": us_devices_list,
        "cookies": us_cookie_dict
    },
    "uk": {
        "devices": uk_devices_list,
        "cookies": uk_cookie_dict
    },
    "de": {
        "devices": de_devices_list,
        "cookies": de_cookie_dict
    }
}

# asin 网页端打开 url
AMAZON_SEARCH_BASE_URL = "https://www.amazon.com/s?rh=p_78:{bbx_asin_list}&rank=asin-scores-asc-rank&searchMethod=CameraSearch"

# 重试策略配置（全局重试次数，包含解析失败）
GLOBAL_RETRY_TIMES = 5  # 整体流程重试次数（含Cookie更新）
STEP_RETRY_TIMES = 5  # 单步请求重试次数
RETRY_DELAY = 1  # 重试延迟（秒）


def get_image_size(image_path: str) -> Optional[Dict[str, int]]:
    """获取图片宽高尺寸"""
    try:
        with Image.open(image_path) as img:
            width, height = img.size
            return {"width": width, "height": height}
    except FileNotFoundError:
        logger.error(f"图片文件未找到: {image_path}")
        return None
    except Exception as e:
        logger.error(f"获取图片尺寸失败 ({image_path}): {e}")
        return None


class AmazonImageSearch:
    def __init__(self, site_name: str):
        '''
        :param site_name:  站点名 如 uk us de
        '''
        # 验证站点合法性
        if site_name not in site_name_secret_dict:
            raise ValueError(f"不支持的站点: {site_name}，支持站点：{list(site_name_secret_dict.keys())}")
        if site_name not in SITE_CONFIG_MAPPER:
            raise ValueError(f"站点 {site_name} 缺少设备或Cookie配置")

        self.site_name = site_name
        self.site_config = site_name_secret_dict[site_name]
        self.site_specific = SITE_CONFIG_MAPPER[site_name]

        # 初始化Cookie和设备信息
        self._update_cookie()  # 初始加载Cookie
        self._update_device()  # 初始加载设备
        self.client_device_id = str(uuid.uuid4())

        logger.info(
            f"客户端初始化完成 - 站点: {self.site_name}, "
            f"随机设备: {self.device_info.get('clientDevice')}, "
            f"clientDeviceId: {self.client_device_id}"
        )

        # 构建请求头（依赖Cookie，初始化时生成）
        self._update_headers()

        self.snap_url = f"{self.site_config['snap_url']}/style-snap/2.0"

    def _update_cookie(self) -> None:
        """更新Cookie（当前固定返回站点对应Cookie，后续可扩展为从数据库读取）"""
        self.cookies = self.site_specific["cookies"].copy()  # 复制一份避免修改原配置
        self.session_id = self.cookies.get("session-id", "")
        logger.info(f"站点 {self.site_name} Cookie已更新（session-id: {self.session_id[:20]}...）")

    def _update_device(self) -> None:
        """更新随机设备信息"""
        devices = self.site_specific["devices"]
        if not devices:
            raise ValueError(f"站点 {self.site_name} 的设备列表为空")
        self.device_info = random.choice(devices)
        logger.debug(f"设备已更新: {self.device_info.get('clientDevice')}")

    def _update_headers(self) -> None:
        """更新请求头（依赖Cookie，Cookie变更后需重新生成）"""
        self.headers = {
            "x-amz-access-token": "",
            "x-amz-lens-session-auth-token": self.cookies.get("session-token", ""),
            "x-amz-lens-session-id": self.session_id,
            "x-amz-lens-ubid": self.cookies.get("ubid-main", ""),
            "accept-encoding": "gzip",
            "user-agent": "okhttp/4.9.1",
        }

    def _generate_auth_params(self) -> Dict[str, str]:
        """生成认证所需的 authtoken 和 ts"""
        ts = str(int(time.time()))
        combined = (
            f"{self.site_config['secret']}{self.site_config['username']}"
            f"{self.site_config['application']}{ts}"
        )
        authtoken = hashlib.sha512(combined.encode("utf-8")).hexdigest()
        return {"ts": ts, "authtoken": authtoken}

    def _build_query_metadata(self, extra_params: Optional[Dict[str, str]] = None) -> str:
        """构建通用的 query_metadata，包含随机设备信息"""
        base_params = {
            "amznSessionId": self.session_id,
            "clientVersion": "30.20.2.100",
            "cardsVersion": "1.0",
            "clientMessageVersion": "1.0",
            "amznDirectedCustomerId": "",
            "clientDeviceId": self.client_device_id,
            "clientId": str(uuid.uuid4()),
            "sourceType": "Photo",
            "ingressSource": "ctp",
            "uiMode": "stylesnap",
            # 注入随机设备信息
            **self.device_info
        }
        if extra_params:
            base_params.update(extra_params)
        return json.dumps(base_params)

    def _parse_response(self, response_json: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """从API响应中解析ASIN列表并构建结果，缺失关键字段视为解析失败"""
        try:
            # 严格检查必须存在的字段
            if "style-snap" not in response_json:
                raise KeyError("响应缺少 'style-snap' 字段")
            if "searchResult" not in response_json["style-snap"]:
                raise KeyError("响应缺少 'style-snap.searchResult' 字段")
            if not response_json["style-snap"]["searchResult"]:
                raise IndexError("'searchResult' 列表为空")

            search_result = response_json["style-snap"]["searchResult"][0]
            bbx_asin_list = search_result.get("bbxAsinList", [])

            if not bbx_asin_list:
                logger.warning("响应中 bbxAsinList 为空")
                return None

            unique_asin_list = list(set(bbx_asin_list))  # 去重asin
            joined_asins = "|".join(unique_asin_list)

            return {
                "is_app": 1,
                "asin_list_app": joined_asins,
                "search_url": AMAZON_SEARCH_BASE_URL.format(bbx_asin_list=joined_asins),
            }
        except (KeyError, IndexError, TypeError) as e:
            logger.error(f"解析响应失败: {e}. 响应内容: {json.dumps(response_json, ensure_ascii=False)}...")
            return None  # 返回None表示解析失败，触发重试

    def _perform_request(self, **kwargs) -> Dict[str, Any]:
        """单步请求的重试逻辑（不更新Cookie，仅处理网络错误）"""
        for attempt in range(STEP_RETRY_TIMES):
            try:
                response = requests.post(**kwargs, timeout=10)
                response.raise_for_status()  # 对非2xx响应抛出HTTPError
                return response.json()
            except JSONDecodeError:
                logger.error(f"请求失败 (第 {attempt + 1}/{STEP_RETRY_TIMES} 次): 响应不是有效的JSON格式")
            except RequestException as e:
                logger.error(f"请求失败 (第 {attempt + 1}/{STEP_RETRY_TIMES} 次): {e}")

            if attempt < STEP_RETRY_TIMES - 1:
                logger.warning(f"将在 {RETRY_DELAY} 秒后重试...")
                time.sleep(RETRY_DELAY)

        raise RequestException(f"单步API请求在 {STEP_RETRY_TIMES} 次尝试后最终失败")

    def _default_search(self, image_path: str) -> Dict[str, Any]:
        """执行默认的图片识别请求（第一步）"""
        logger.info(f"开始默认识别 (站点: {self.site_name}, 图片: {image_path}), 站点链接：{self.snap_url}")

        try:
            with open(image_path, "rb") as f:
                image_data = f.read()
        except FileNotFoundError:
            logger.error(f"无法读取图片文件: {image_path}")
            raise

        auth_params = self._generate_auth_params()
        query_metadata = self._build_query_metadata({"orientation": "-1"})

        files = {
            "application": (None, self.site_config['application']),
            "query_metadata": (None, query_metadata),
            "authtoken": (None, auth_params['authtoken']),
            # "authtoken": (None, ''), # 为空失请求失败 测试重试功能
            "lang": (None, "en_US"),
            "username": (None, self.site_config['username']),
            "ts": (None, auth_params['ts']),
            "file": ("image.jpg", image_data, "image/jpeg"),
        }
        return self._perform_request(url=self.snap_url, files=files, headers=self.headers)

    def _full_image_search(self, query_id: str, image_path: str) -> Dict[str, Any]:
        """执行全图识别请求（第二步）"""
        logger.info(f"开始全图识别 (Query ID: {query_id[:10]}...)")

        image_size = get_image_size(image_path)
        if not image_size:
            raise ValueError("无法获取图片尺寸，无法进行全图搜索")

        # 生成随机裁剪框
        offset = random.randint(0, 2)
        bounding_box = {
            "tlx": max(0, offset),
            "tly": max(0, offset),
            "brx": max(image_size["width"] - offset, max(0, offset) + 1),
            "bry": max(image_size["height"] - offset, max(0, offset) + 1),
            "imh": image_size["height"],
            "imw": image_size["width"]
        }

        auth_params = self._generate_auth_params()
        query_metadata = self._build_query_metadata()

        form_data = {
            "mainQueryId": (None, query_id),
            "uiMode": (None, "stl_bbx_reformulation"),
            "application": (None, self.site_config['application']),
            "query_metadata": (None, query_metadata),
            "authtoken": (None, auth_params['authtoken']),
            "inputBoundingBox": (None, json.dumps(bounding_box)),
            "imageHash": (None, ""),
            "lang": (None, "en_US"),
            "username": (None, self.site_config['username']),
            "ts": (None, auth_params['ts']),
        }

        return self._perform_request(url=self.snap_url, files=form_data, headers=self.headers)

    def search(self, image_path: str, search_mode: str = "default") -> Dict[str, Any]:
        """
        执行图片搜索（包含全局重试逻辑，解析失败时更新Cookie重试）
        :param image_path: 本地图片文件路径
        :param search_mode: 搜索模式 ('default' 或 'full_image')
        :return: 包含搜索结果的字典
        """
        base_result = {
            "is_web": 0, "is_app": 0, "asin_list_web": "", "asin_list_app": "",
            "asin_list_join": "", "site_name": self.site_name,
            "search_url": None, "mode": search_mode,
            "retry_count": 0  # 记录重试次数
        }

        # 全局重试逻辑（包含Cookie更新）
        for global_attempt in range(GLOBAL_RETRY_TIMES):
            try:
                # 检查Cookie有效性（每次重试前确认）
                if not self.session_id:
                    logger.warning("Cookie中缺少'session-id'，尝试更新Cookie...")
                    self._update_cookie()
                    self._update_headers()  # 同步更新请求头

                # 步骤1: 执行默认搜索（全图模式依赖此步骤的query_id）
                default_response = self._default_search(image_path)

                # 处理默认模式
                if search_mode == "default":
                    parsed_result = self._parse_response(default_response)
                    if parsed_result:  # 解析成功才返回
                        base_result.update(parsed_result)
                        base_result["asin_list_join"] = parsed_result["asin_list_app"]
                        base_result["retry_count"] = global_attempt
                        return base_result
                    else:
                        # 解析失败，准备重试
                        raise ValueError("默认模式响应解析失败，触发重试")

                # 处理全图模式
                elif search_mode == "full_image":
                    query_id = default_response.get("queryId")
                    if not query_id:
                        raise ValueError("默认识别未返回queryId，无法进行全图搜索")

                    full_image_response = self._full_image_search(query_id, image_path)
                    parsed_result = self._parse_response(full_image_response)
                    if parsed_result:  # 解析成功才返回
                        base_result.update(parsed_result)
                        base_result["asin_list_join"] = parsed_result["asin_list_app"]
                        base_result["retry_count"] = global_attempt
                        return base_result
                    else:
                        # 解析失败，准备重试
                        raise ValueError("全图模式响应解析失败，触发重试")

                else:
                    logger.error(f"不支持的搜索模式: {search_mode}")
                    return base_result

            except Exception as e:
                logger.error(f"第 {global_attempt + 1}/{GLOBAL_RETRY_TIMES} 次尝试失败: {e}")

                # 最后一次重试失败则返回基础结果
                if global_attempt == GLOBAL_RETRY_TIMES - 1:
                    base_result["error"] = str(e)
                    return base_result

                # 非最后一次重试：更新Cookie和设备，延迟后重试
                logger.info("准备更新Cookie和设备信息后重试...")
                self._update_cookie()  # 更新Cookie（当前固定站点Cookie，后续可从数据库读取）
                self._update_device()  # 更新设备信息
                self._update_headers()  # 同步更新请求头
                self.client_device_id = str(uuid.uuid4())  # 更新客户端ID
                logger.info(f"重试准备完成（下次是第 {global_attempt + 2} 次尝试）")
                time.sleep(RETRY_DELAY)

        return base_result


if __name__ == "__main__":
    # 模拟图片路径
    image_file_path = "temp_img/B0BYNB2J6W.jpg"

    try:
        # 测试不同站点
        for site in ["us","uk","de"]:
            logger.info(f"\n{'=' * 40}")
            logger.info(f"开始测试站点: {site}")
            logger.info(f"{'=' * 40}")
            client = AmazonImageSearch(site_name=site)

            # 测试默认识别模式
            logger.info("\n--- 测试默认识别模式 ---")
            default_result = client.search(image_file_path, search_mode="default")
            logger.info(f"默认模式结果: {json.dumps(default_result, ensure_ascii=False, indent=2)}")

            # # 测试全图识别模式
            logger.info("\n--- 测试全图识别模式 ---")
            full_image_result = client.search(image_file_path, search_mode="full_image")
            logger.info(f"全图模式结果: {json.dumps(full_image_result, ensure_ascii=False, indent=2)}")

    except ValueError as e:
        logger.error(f"初始化失败: {e}")
    except Exception as e:
        logger.error(f"执行过程中发生错误: {e}")