import os
import time
import threading
import queue
import random
import json
from datetime import datetime
from typing import List, Dict
from loguru import logger
import pandas as pd
from app_img_search_api_new import AmazonImageSearch  # 导入客户端类
from amazon_configs import (
    site_name_secret_dict,
    us_devices_list,
    uk_devices_list,
    de_devices_list
)
# 日志目录配置
LOG_DIR = "log"  # 日志文件夹名称

# 判断日志目录是否存在，不存在则创建
if not os.path.exists(LOG_DIR):
    os.makedirs(LOG_DIR, exist_ok=True)  # exist_ok=True 避免目录已存在时报错

# 测试配置
TEST_FOLDER = "temp_img"  # 图片文件夹路径
REQUESTS_PER_IMAGE = 1  # 每张图片每个站点的请求次数（每种模式各执行一次）
THREAD_COUNT = 30  # 线程数量
RESULT_EXCEL = "multi_mode_test_results3.xlsx"  # 结果保存文件
ALL_SITES = ["us", "uk", "de"]  # 要测试的三个站点
ALL_MODES = ["default", "full_image"]  # 两种搜索模式

# 站点设备列表映射
SITE_DEVICES = {
    "us": us_devices_list,
    "uk": uk_devices_list,
    "de": de_devices_list
}

# 线程安全组件
task_queue = queue.Queue()
result_lock = threading.Lock()
test_results = []


def worker():
    """线程工作函数：先执行默认模式，再执行全图模式，记录两种模式结果"""
    thread_id = threading.current_thread().ident
    thread_name = threading.current_thread().name

    while not task_queue.empty():
        try:
            # 从队列获取任务：图片路径、请求序号、站点
            image_path, request_idx, site_name = task_queue.get(block=False)
            image_name = os.path.basename(image_path)
            logger.info(
                f"[{thread_name}] 开始处理 {image_name} 第 {request_idx} 次请求 "
                f"（站点：{site_name}，两种模式）"
            )

            # 初始化客户端（同一客户端实例，确保设备信息一致）
            client = AmazonImageSearch(site_name=site_name)
            device_info = client.device_info  # 获取设备信息（两种模式共用）
            device_str = (
                f"设备: {device_info.get('clientDevice')}, "
                f"厂商: {device_info.get('deviceManufacturer')}, "
                f"系统版本: {device_info.get('clientDeviceVersion')}"
            )
            logger.debug(f"[{thread_name}] {device_str}")

            # 依次执行两种模式
            for mode in ALL_MODES:
                try:
                    # 记录单次模式开始时间
                    mode_start = time.time()

                    # 执行对应模式的搜索
                    result_data = client.search(
                        image_path=image_path,
                        search_mode=mode
                    )

                    # 计算单次模式耗时
                    duration = round(time.time() - mode_start, 4)
                    save_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")

                    # 锁定并保存结果（保持原有数据格式，新增mode字段）
                    with result_lock:
                        test_results.append({
                            "image": image_name,
                            "request_index": request_idx,  # 同一请求序号关联两种模式
                            "mode": mode,  # 新增：标记当前模式
                            "duration": duration,
                            "success": result_data.get("is_app", 0),
                            "result_url": result_data.get("search_url", "None"),
                            "thread_id": thread_id,
                            "is_web": result_data.get("is_web", 0),
                            "is_app": result_data.get("is_app", 0),
                            "asin_list_web": result_data.get("asin_list_web", ""),
                            "asin_list_app": result_data.get("asin_list_app", ""),
                            "asin_list_join": result_data.get("asin_list_join", ""),
                            "site_name": site_name,
                            "device_info": json.dumps(device_info, ensure_ascii=False),
                            # "device_summary": f"{device_info.get('clientDevice')} ({device_info.get('clientDeviceVersion')})",
                            "save_time": save_time
                        })
                    logger.info(f"[{thread_name}] {mode}模式完成（耗时{duration}秒）")

                except Exception as e:
                    logger.error(f"[{thread_name}] {mode}模式处理出错: {str(e)}")
                    # 记录模式失败结果
                    with result_lock:
                        test_results.append({
                            "image": image_name,
                            "request_index": request_idx,
                            "mode": mode,
                            "duration": 0,
                            "success": 0,
                            "result_url": "None",
                            "thread_id": thread_id,
                            "is_web": 0,
                            "is_app": 0,
                            "asin_list_web": "",
                            "asin_list_app": "",
                            "asin_list_join": "",
                            "site_name": site_name,
                            "device_info": json.dumps(device_info, ensure_ascii=False),
                            # "device_summary": f"{device_info.get('clientDevice')} ({device_info.get('clientDeviceVersion')})",
                            "save_time": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                            # "error": str(e)
                        })

        except queue.Empty:
            break
        except Exception as e:
            logger.error(f"[{thread_name}] 整体任务出错: {str(e)}")
        finally:
            task_queue.task_done()


def collect_image_paths(folder: str) -> List[str]:
    """收集文件夹中的图片路径（仅支持常见图片格式）"""
    image_extensions = ('.png', '.jpg', '.jpeg', '.gif', '.bmp')
    if not os.path.exists(folder):
        logger.error(f"图片文件夹不存在: {folder}")
        return []
    return [
        os.path.join(folder, f)
        for f in os.listdir(folder)
        if f.lower().endswith(image_extensions)
    ]


def init_task_queue(image_paths: List[str]):
    """初始化任务队列：包含所有站点、所有图片、所有请求次数（一次任务对应两种模式）"""
    for site in ALL_SITES:
        for request_idx in range(1, REQUESTS_PER_IMAGE + 1):
            for image_path in image_paths:
                task_queue.put((image_path, request_idx, site))
    total_tasks = task_queue.qsize()
    logger.info(
        f"任务队列初始化完成 - 总任务数: {total_tasks} "
        f"（每个任务包含{len(ALL_MODES)}种模式），"
        f"站点数: {len(ALL_SITES)}, 图片数: {len(image_paths)}, "
        f"单图单站请求数: {REQUESTS_PER_IMAGE}"
    )


def save_results_to_excel(results: List[Dict]):
    """将结果写入Excel，新增mode字段区分模式"""
    if not results:
        logger.warning("无结果可保存")
        return

    # 转换为DataFrame，确保字段顺序（新增mode字段）
    df = pd.DataFrame(results)[[
        "image", "request_index", "mode", "duration", "success", "result_url",
        "thread_id", "is_web", "is_app", "asin_list_web", "asin_list_app",
        "asin_list_join", "site_name", "device_info", "save_time"
    ]]

    # 保存到Excel
    df.to_excel(RESULT_EXCEL, index=False, engine="openpyxl")
    logger.info(f"结果已保存到 {RESULT_EXCEL}")


def print_summary(results: List[Dict]):
    """打印测试总结（按站点和模式分组统计）"""
    if not results:
        return

    total = len(results)
    total_duration = sum(r["duration"] for r in results)
    logger.info("\n===== 测试总结 =====")
    logger.info(f"总请求数: {total}（含{len(ALL_MODES)}种模式），"
                f"总耗时: {total_duration:.2f}秒，"
                f"平均耗时: {total_duration / total:.4f}秒")

    # 按站点分组统计
    for site in ALL_SITES:
        site_results = [r for r in results if r["site_name"] == site]
        site_total = len(site_results)
        if site_total == 0:
            continue

        logger.info(f"\n----- 站点: {site} 统计 -----")
        logger.info(f"总请求数: {site_total}，平均耗时: {sum(r['duration'] for r in site_results) / site_total:.4f}秒")

        # 按模式分组统计
        for mode in ALL_MODES:
            mode_results = [r for r in site_results if r["mode"] == mode]
            mode_total = len(mode_results)
            if mode_total == 0:
                continue

            mode_success = sum(1 for r in mode_results if r["is_app"] == 1)
            mode_success_rate = (mode_success / mode_total) * 100 if mode_total else 0
            logger.info(
                f"模式 {mode}: "
                f"请求数 {mode_total}, 成功数 {mode_success}, "
                f"成功率 {mode_success_rate:.2f}%, "
                f"平均耗时 {sum(r['duration'] for r in mode_results) / mode_total:.4f}秒"
            )

        # 设备多样性统计
        # devices_used = {r["device_summary"] for r in site_results}
        # logger.info(f"使用设备种类: {len(devices_used)} 种")


if __name__ == "__main__":
    # 初始化日志
    log_file = os.path.join(LOG_DIR, "multi_mode_stability_test.log")
    logger.add(log_file,rotation="10 MB",level="INFO",encoding="utf-8")
    # 验证所有站点的合法性
    for site in ALL_SITES:
        if site not in site_name_secret_dict:
            logger.error(f"不支持的站点: {site}，支持站点: {list(site_name_secret_dict.keys())}")
            exit(1)
        if not SITE_DEVICES.get(site, []):
            logger.warning(f"站点 {site} 的设备列表为空，可能影响测试结果")

    # 准备资源
    image_paths = collect_image_paths(TEST_FOLDER)
    if not image_paths:
        logger.error("未找到图片，退出测试")
        exit(1)
    init_task_queue(image_paths)

    # 启动多线程
    start_time = time.time()
    threads = []
    for i in range(THREAD_COUNT):
        t = threading.Thread( target=worker, name=f"Worker-{i + 1}")
        threads.append(t)
        t.start()
        logger.info(f"启动线程: {t.name}")

    # 等待所有任务完成
    task_queue.join()
    for t in threads:
        t.join()

    total_time = time.time() - start_time
    logger.info(f"所有任务完成，总耗时: {total_time:.2f}秒")

    # 保存结果并打印总结
    save_results_to_excel(test_results)
    print_summary(test_results)