# -*- coding: utf-8 -*-
import time
import logging
import aiohttp
import scrapy
from scrapy import signals
from amazon_spider.utils.random_ssl import sslgen

logger = logging.getLogger(__name__)


class AiohttpMiddleware:
    """
    scrapy timeout就用aiohttp试试
    用于解决一些蜜汁bug
    """

    def __init__(self, timeout=10):
        self.timeout = timeout

    @classmethod
    def from_crawler(cls, crawler):
        # This method is used by Scrapy to create your spiders.
        timeout = crawler.settings.get('DOWNLOAD_TIMEOUT', 10)
        s = cls(timeout)
        crawler.signals.connect(s.spider_opened, signal=signals.spider_opened)
        return s

    def spider_opened(self, spider):
        spider.logger.info(
            "Spider  %s opened middleware: %s" % (spider.name, self.__class__.__name__)
        )

    async def process_request(self, request: scrapy.Request, spider):
        from scrapy.http.headers import Headers
        # logger.info("进来的meta是 {0}".format(request.meta))
        if request.meta.get("use_aiohttp", False):
            proxies = {
                'http': request.meta.get('proxy'),
                'https': request.meta.get('proxy')
            }
            logger.debug("使用aiohttp进行尝试")
            # url = request.url
            headers = Headers(request.headers or {}, encoding='utf-8').to_unicode_dict()
            start_time = time.time()
            async with aiohttp.ClientSession() as session:
                async with session.request(
                        request.method,
                        request.url,
                        data=request.body,
                        headers=headers,
                        proxy=proxies.get("https"),
                        timeout=self.timeout,
                        cookies=request.cookies,
                        ssl=sslgen()) as resp:
                    html: bytes = await resp.read()
                    end_time = time.time()  # 记录结束时间
                    response_time = end_time - start_time  # 计算响应时间
                    logger.info(f"aiohttp Response time: {response_time} seconds")
                    stats = spider.crawler.stats
                    stats.inc_value("downloader/request_count")
                    stats.inc_value(f"downloader/request_method_count/{request.method}")
                    return scrapy.http.HtmlResponse(
                        url=request.url,
                        status=resp.status,
                        headers=request.headers,
                        body=html,
                        request=request,
                        # encoding=resp.get_encoding(),
                        encoding="utf-8",
                    )

    def process_response(self, request, response, spider):
        return response

    def process_exception(self, request, exception, spider):
        logging.info(f"middleware error {spider.r_utils(request)} {exception}")
        return None