import pandas as pd
import numpy as np
import math
from sklearn.linear_model import LinearRegression

def estimate_by_interpolation(df_known, min_rank, max_rank):
    df_k = df_known.set_index('rank').sort_index()
    idx = range(min_rank, max_rank+1)
    df_full = df_k.reindex(idx)
    df_full['orders'] = df_full['orders'].interpolate()
    df_full = df_full.reset_index().rename(columns={'index':'rank'})
    df_full['orders_day'] = df_full['orders'].apply(lambda x: math.ceil(x/30))
    return df_full

def estimate_by_powerlaw(df_known, min_rank, max_rank):
    ranks = np.log(df_known['rank'].values).reshape(-1,1)
    orders = np.log(df_known['orders'].values)
    model = LinearRegression().fit(ranks, orders)
    a = np.exp(model.intercept_)
    b = -model.coef_[0]
    ranks_full = np.arange(min_rank, max_rank+1)
    orders_pred = a * (ranks_full ** -b)
    df_pred = pd.DataFrame({
        'rank': ranks_full,
        'orders_day': np.ceil(orders_pred/30)
    })
    return df_pred

# 假设 df_known 是 5月已知的样本：(rank, orders)
df_known = pd.DataFrame({
    'rank':[10000,11000,12000,13000,14000,15000,16000],
    'orders':[3240,3000,2790,2640,2460,2340,2220]
})

df_interp = estimate_by_interpolation(df_known, 10000, 20000)
df_power  = estimate_by_powerlaw(df_known,    10000, 20000)

print("=== 插值预测（前10行） ===")
print(df_interp.head(30))
print("\n=== 幂律预测（前10行） ===")
print(df_power.head(30))
