import { localCache } from '@/utils/storage'
import { defineStore } from 'pinia'
import type { RouteLocationNormalized } from 'vue-router'

type ICallbackType = (...args: any[]) => any

export const useAppStore = defineStore('app', {
  id: 'app',
  state: () => {
    return {
      // 上一级路由
      lastRoute: {} as RouteLocationNormalized,

      // 移动端下拉刷新 0不显示  1下拉中 2准备刷新  3 刷新中 4 刷新完成
      refreshStatus: 0,

      // 移动端是否弹出键盘
      isShowKeyBoard: false,

      // 记忆页面滚动位置,key:页面地址 value:Y轴滚动
      cacheScrollPage: {} as { [key: string]: number },

      // 应用内是否存在全局覆盖层，如果存在，返回上一级改成退出全局覆盖层
      // 如：图片预览，评论弹窗
      fullScreenInstance: {
        state: 0,
        instance: null, // 打开的示例
        close: undefined as ICallbackType | undefined, // 关闭方法
        curUrl: '', // 页面Url
      },

      // 上一次公告弹出时间
      announcementTime: 0,
    }
  },
  actions: {
    updateLastRoute(route: RouteLocationNormalized) {
      this.lastRoute = route
    },

    updateRefreshStatus(status: number) {
      this.refreshStatus = status
    },

    //软键盘收起的事件处理
    handleShowKeyBoard(device = '') {
      // ios处理
      if (device === 'ios') {
        document.body.addEventListener('focusin', () => {
          this.isShowKeyBoard = true
        })
        document.body.addEventListener('focusout', () => {
          this.isShowKeyBoard = false
        })
      } else if (device === 'android') {
        const originalHeight = document.documentElement.clientHeight || document.body.clientHeight
        window.onresize = () => {
          //键盘弹起与隐藏都会引起窗口的高度发生变化
          const resizeHeight = document.documentElement.clientHeight || document.body.clientHeight
          if (resizeHeight - 0 < originalHeight - 0) {
            this.isShowKeyBoard = true
          } else {
            this.isShowKeyBoard = false
          }
        }
      }
    },

    /**
     * 页面滚动位置
     */
    setScrollPage(url: string, top: number) {
      this.cacheScrollPage[url] = top
    },
    getScrollPage(url: string) {
      return this.cacheScrollPage[url]
    },

    /**
     * 全局覆盖层
     * @returns 返回false，说明 不要返回上一页，而是关闭全局覆盖层
     */
    closeFullScreen() {
      if (this.fullScreenInstance.state) {
        this.fullScreenInstance.close && this.fullScreenInstance.close()

        this.clearFullScreenInstance()
        return false
      }
      return true
    },
    // 清除全屏模态框实例
    clearFullScreenInstance() {
      this.fullScreenInstance = {
        state: 0,
        curUrl: '',
        instance: null,
        close: undefined,
      }
    },
    // 插入全屏模态框示例
    updateFullScreenInstance(curUrl: string, instance: any, closeCallback: any) {
      this.fullScreenInstance = {
        state: 1,
        instance,
        curUrl,
        close: closeCallback,
      }
    },

    /**
     * 公告时间
     */
    showAnnouncement() {
      localCache.setCache('announcementTime', Date.now())
    },
    getAnnouncement() {
      this.announcementTime = localCache.getCache<number>('announcementTime') || 0
      return this.announcementTime
    },
  },
})
