import { ArticleTypeEnum, ReleaseStatusTypeEnum, SendTypeEnum } from '@/constants'
import UploadFile from '@/components/common/UploadFile/index.vue'
import { useResetData } from '@/hooks'
import type { AddOrUpdatePostDto } from '@/api'

export default defineComponent(
  (_, { expose }) => {
    const [form, resetForm] = useResetData<AddOrUpdatePostDto>({
      title: '',
      content: '',
      faceUrl: '',
      imgUrl: '',
      releaseStatus: ReleaseStatusTypeEnum.PUBLISH,
      type: ArticleTypeEnum.POST,
      sendType: SendTypeEnum.IMMEDIATE,
      sendTime: '',
    })
    const formRef = ref<InstanceType<typeof ElForm>>()
    const rules = {
      title: [{ required: true, message: '请输入帖子标题', trigger: 'blur' }],
      content: [{ required: true, message: '请输入帖子内容', trigger: 'blur' }],
      sendType: [{ required: true, message: '请选择发布类型', trigger: 'blur' }],
      sendTime: [{ required: true, message: '请选择发布时间', trigger: 'blur' }],
    }

    const transformForm = (releaseStatus: ReleaseStatusTypeEnum) => {
      return {
        ...form.value,
        releaseStatus,
        faceUrl: form.value.imgUrl?.split(',').filter(Boolean)[0] || '',
      }
    }

    const getValidatedFormData = async (releaseStatus: ReleaseStatusTypeEnum) => {
      try {
        await formRef.value?.validate()
        return transformForm(releaseStatus)
      } catch (error) {
        console.log(error)
        ElMessage.warning('请检查输入内容')
        return null
      }
    }

    const resetFields = () => {
      formRef.value?.resetFields()
      resetForm()
    }

    expose({
      getValidatedFormData,
      resetFields,
    })
    return () => (
      <div>
        <el-form
          ref={formRef}
          model={form.value}
          label-width="auto"
          label-position="right"
          rules={rules}
        >
          <el-form-item label="标题" prop="title">
            <el-input
              v-model={form.value.title}
              placeholder="请输入帖子标题"
              maxlength={30}
              show-word-limit
            />
          </el-form-item>
          <el-form-item label="内容" prop="content">
            <el-input
              v-model={form.value.content}
              type="textarea"
              placeholder="良言良语，快乐无限；恶语恶言，伤心伤肝。请简述发帖内容，或点击长文章编辑格式"
              rows={6}
              maxlength={255}
              show-word-limit
              class="content-input"
            />
          </el-form-item>
          <el-form-item label="图片" prop="faceUrl">
            {/* @ts-ignore */}
            <UploadFile v-model={form.value.faceUrl} />
          </el-form-item>
          <el-form-item label="发布类型" prop="sendType">
            <el-radio-group v-model={form.value.sendType} class="radio-group">
              <el-radio value={SendTypeEnum.IMMEDIATE} class="radio-item immediate">
                立即发布
              </el-radio>
              <el-radio value={SendTypeEnum.SCHEDULED} class="radio-item scheduled">
                定时发布
              </el-radio>
            </el-radio-group>
          </el-form-item>
          {form.value.sendType === SendTypeEnum.SCHEDULED && (
            <el-form-item label="发布时间" prop="sendTime">
              <el-date-picker
                class="w-full"
                v-model={form.value.sendTime}
                type="datetime"
                placeholder="请选择发布时间"
                value-format="X"
              />
            </el-form-item>
          )}
        </el-form>
      </div>
    )
  },
  {
    name: 'PostForm',
  },
)
