/**
 * 日期格式化
 * @param value 日期
 * @param format 格式化 'YYYY-MM-DD HH:mm:ss'
 * @param useAlias 是否使用别名，如：3小时前
 * @returns
 */
export function time_format(value?: string | number, format = 'YYYY-MM-DD', useAlias = false) {
  if (!value) {
    return '-'
  }
  if (useAlias) {
    return formatDateAlias(value, format)
  }
  if (Number.isNaN(Number(value))) {
    // 字符串类型日期：YYYY-MM-DD hh:mm:ss
    return formatDate(value, format)
  }
  // 时间戳日期格式: (秒 是13位以上)
  if (value.toString().length >= 13) {
    return formatDate(value, format)
  }
  return formatDate(Number(value) * 1000, format)
}

// 将日期格式化成对应的格式
export const formatDate = function (str: string | number, format = 'YYYY-MM-DD HH:mm', shortStr = false) {
  // 兼容苹果
  if (typeof str === 'string') {
    str = str.replace(/-/g, '/')
  }

  let date
  if (!str) {
    date = new Date()
  } else {
    date = new Date(str)
  }

  const now = new Date()
  // 是否去除年显示
  let formatStr = format
  const nowYear = now.getFullYear()
  const year = date.getFullYear()
  if (shortStr && nowYear === year) {
    formatStr = formatStr.replace('YYYY-', '')
  }

  // 是否去除月日显示
  const nowMonth = new Date().getMonth()
  const month = date.getMonth()
  const nowDay = new Date().getDate()
  const day = date.getDate()
  if (shortStr && nowYear === year && nowMonth === month && nowDay === day) {
    formatStr = formatStr.replace('MM-DD ', '')
  }

  const y = date.getFullYear()
  let MM: number | string = date.getMonth() + 1
  MM = MM < 10 ? '0' + MM : MM
  let d: number | string = date.getDate()
  d = d < 10 ? '0' + d : d
  let h: number | string = date.getHours()
  h = h < 10 ? '0' + h : h
  let m: number | string = date.getMinutes()
  m = m < 10 ? '0' + m : m
  let s: number | string = date.getSeconds()
  s = s < 10 ? '0' + s : s

  const time = formatStr
    .replace('YYYY', y + '')
    .replace('MM', MM + '')
    .replace('DD', d + '')
    .replace('HH', h + '')
    .replace('mm', m + '')
    .replace('ss', s + '')
  return time
}
// 将日期变成对应的别名
export const formatDateAlias = function (value: string | number, format = 'YYYY-MM-DD HH:mm:ss') {
  // 兼容苹果
  if (typeof value === 'string') {
    value = value.replace(/-/g, '/')
  }
  if (typeof value === 'number' && value.toString().length < 13) {
    value = Number(value) * 1000
  }
  const now = Date.now()
  const time = new Date(value).getTime()
  let unit = '后'
  if (now > time) {
    unit = '前'
  }

  // x秒前/后
  let alias = '秒'
  const intervalTime = Math.abs(now - time)
  const seconds = Math.floor(intervalTime / 1000) + 1
  alias = seconds + alias
  if (seconds < 60) {
    return alias + unit
  }

  // x分前/后
  alias = '分'
  const minutes = Math.floor(seconds / 60) + 1
  alias = minutes + alias
  if (minutes < 60) {
    return alias + unit
  }

  // 今天
  const date = new Date(value)
  const nowDate = new Date()
  const days = date.getDate()
  const month = date.getMonth()
  const year = date.getFullYear()
  const nowDays = nowDate.getDate()
  const nowMonth = nowDate.getMonth()
  const nowYear = nowDate.getFullYear()
  if (days === nowDays && month === nowMonth && year === nowYear) {
    return '今天 ' + formatDate(value, 'HH:mm')
  }

  return formatDate(value, format, true)
}

/**
 * 给一个时期，获取周的开始和结束
 * @param date
 */
export const getWeek = function (date: Date, format?: string) {
  const weekNum = date.getDay()
  const week = []
  const dayTime = 1000 * 60 * 60 * 24
  for (let i = 1; i <= 7; i++) {
    const value = new Date(date.getTime() - dayTime * (weekNum - i))
    if (format) {
      week.push(time_format(value.getTime(), format))
    } else {
      week.push(value)
    }
  }

  return week
}

/**
 * 给一个时间分为，获取相隔几天
 */
export const getRangeNum = function (startTime = 0, endTime = 0) {
  let rangeTime = endTime - startTime
  if (startTime.toString().length < 13) {
    rangeTime = (endTime - startTime) * 1000
  }
  return Math.floor(rangeTime / (60 * 60 * 24 * 1000))
}

/**
 * 给一个时间范围，获取对应时间范围
 * @param date
 */
export const getRangeByStartEnd = function (startTime = 0, endTime = 0, format?: string) {
  const rangeNum = getRangeNum(startTime, endTime)
  if (!rangeNum) return []

  const arr = [time_format(startTime, format)]
  for (let i = 1; i < rangeNum; i++) {
    const date = time_format(startTime + 60 * 60 * 24 * i, format)
    arr.push(date)
  }
  arr.push(time_format(endTime, format))
  return arr
}
