import type { RouterScrollBehavior } from 'vue-router'

// 滚动位置存储 Map
const scrollPositionMap = new Map<string, number>()

/**
 * 保存滚动位置
 * @param path 路由路径
 * @param position 滚动位置
 */
export function saveScrollPosition(path: string, position: number): void {
  scrollPositionMap.set(path, position)
}

/**
 * 获取滚动位置
 * @param path 路由路径
 */
export function getScrollPosition(path: string): number | undefined {
  return scrollPositionMap.get(path)
}

/**
 * 清除滚动位置
 * @param path 路由路径（可选，不传则清除所有）
 */
export function clearScrollPosition(path?: string): void {
  if (path) {
    scrollPositionMap.delete(path)
  } else {
    scrollPositionMap.clear()
  }
}

/**
 * 路由滚动行为配置
 */
export const scrollBehavior: RouterScrollBehavior = (to, from, savedPosition) => {
  return new Promise((resolve) => {
    // 1. 如果有浏览器保存的位置（前进/后退），优先使用
    if (savedPosition) {
      resolve(savedPosition)
      return
    }

    // 2. 如果有锚点，滚动到锚点
    if (to.hash) {
      resolve({
        el: to.hash,
        behavior: 'smooth', // 平滑滚动
      })
      return
    }

    // 3. 检查是否有保存的滚动位置
    const savedScrollY = scrollPositionMap.get(to.fullPath)
    if (savedScrollY !== undefined) {
      resolve({
        top: savedScrollY,
        behavior: 'smooth',
      })
      return
    }

    // 4. 默认滚动到顶部
    resolve({ top: 0 })
  })
}

// 导出默认配置
export default scrollBehavior
