import * as XLSX from 'xlsx'

/**
 * 导出 Excel
 * @param {Object[]} data 数据源
 * @param {Object[]} columns 列配置
 * @param {String} fileName 文件名
 */

export interface ExportColumn<T, K extends keyof T = keyof T> {
  title: string // 表头中文名
  key: K // 数据源中后端返回的字段名
  width?: number // 列宽
  formatter?: (value: T[K]) => any // 自定义话格式函数 比如后端返回 0 1 转换为 否 是
}

export interface ExportOptions<T> {
  data: T[]
  columns: ExportColumn<T>[]
  fileName?: string
  sheetName?: string
}

export function exportExcel<T>({
  data,
  columns,
  fileName = '导出数据',
  sheetName = 'Sheet1',
}: ExportOptions<T>) {
  if (!data || !data.length) return ElMessage.warning('没有可导出的数据')

  const header = columns.map((col) => col.title)

  const rows = data.map((row) =>
    columns.map((col) => {
      const value = row[col.key]
      return col.formatter ? col.formatter(value) : value
    }),
  )
  // 生成 worksheet
  const worksheet = XLSX.utils.aoa_to_sheet([header, ...rows])

  // 设置列宽
  worksheet['!cols'] = columns.map((col) => ({
    wch: col.width || 15,
  }))

  // 生成 workbook
  const workbook = XLSX.utils.book_new()
  XLSX.utils.book_append_sheet(workbook, worksheet, sheetName)

  // 导出
  XLSX.writeFile(workbook, `${fileName}.xlsx`)
}
