import { defineStore } from 'pinia'
import { localCache } from '@/utils/storage'
import {
  getChatPeople,
  getPendingReviewCount,
  getUserByCode,
  type IPendingCountMap,
  type IUserInfo,
} from '@/api'

export const useUserStore = defineStore({
  id: 'user',
  state: () => {
    return {
      id: '',
      name: '',
      isManager: false,
      isOfficial: false, // 是否官方
      userInfo: {} as IUserInfo,
      expireTime: 0, // 过期时间

      // 私信：消息维度
      unReadNotice: 0,

      // 管理员未处理数量
      unDoneNum: {} as IPendingCountMap,

      // 是否已被封禁
      isBanned: false,

      // 没有权限名单
      // noAuthList: ['yuxueqin']
      noAuthList: [] as string[],
    }
  },
  getters: {
    getUserInfo(state) {
      if (state.userInfo && Object.keys(state.userInfo).length) return state.userInfo
      const userInfo = localCache.getCache<IUserInfo>('userInfo')
      if (userInfo) {
        state.userInfo = userInfo
        state.id = userInfo.userId
        state.name = userInfo.username
        state.isManager = userInfo.hasManagerBtn
        state.isBanned = !!userInfo.isBan
        state.isOfficial = !!userInfo.isOfficialAccount
      }
      return userInfo
    },
    getExpireTime(state) {
      if (state.expireTime) return state.expireTime
      const expireTime = localCache.getCache<number>('expireTime')
      if (expireTime) {
        state.expireTime = expireTime
      }
      return expireTime
    },

    isNoAuth(state) {
      return state.noAuthList.includes(state.userInfo.account)
    },

    manageArea(state) {
      return (state.userInfo.manageArea || '').split(',') || []
    },
  },
  actions: {
    // 判断是否拥有管理权限，传入地区，不传则不判断地区
    hasManagerPermission(area?: string) {
      if (!this.isManager) return false
      if (area) {
        return this.manageArea.includes(area)
      }
      return true
    },

    setUserInfo(userInfo: IUserInfo) {
      const expireTime = Date.now() + 3600 * 1000 * 24 * 4 // 4个小时
      this.id = userInfo.userId
      this.name = userInfo.username
      this.isManager = userInfo.hasManagerBtn
      this.isOfficial = !!userInfo.isOfficialAccount
      this.isBanned = !!userInfo.isBan
      this.userInfo = userInfo
      this.expireTime = expireTime
      localCache.setCache('userInfo', userInfo)
      localCache.setCache('expireTime', expireTime)
    },

    /**
     * 通过code获取用户数据
     * @param isCodeLogin code来自系统生成还是企业微信 0 来自企业微信，1来自系统生成,默认来自企业微信
     */
    async getUserInfoByCode(code: string, isCodeLogin: number = 0, cutEmail?: string) {
      try {
        const res = await getUserByCode(code, isCodeLogin, cutEmail)
        this.setUserInfo(res.data)
      } catch (error) {
        return Promise.reject(error)
      }
    },

    // 获取未读消息
    async refreshNoticeStatus() {
      const res = await getChatPeople()
      this.unReadNotice = res.data.dialogList.reduce((pre, cur) => {
        return cur.un_read_count + pre
      }, 0)
    },

    // 获取管理员未处理数量
    async getPendingReviewCount() {
      if (!this.isManager) return
      const res = await getPendingReviewCount()
      const obj: IPendingCountMap = {}
      res.data.forEach((v) => {
        if (v.type === 1) {
          obj.auditCount = v
        } else if (v.type === 2) {
          obj.topCount = v
        } else if (v.type === 3) {
          obj.reportCount = v
        } else if (v.type === 4) {
          obj.unbanCount = v
        }
      })
      this.unDoneNum = obj
    },
  },
})
