import type { RouterScrollBehavior } from 'vue-router'

// 滚动位置存储 Map
const scrollPositionMap = new Map<string, number>()

/**
 * 保存滚动位置
 * @param path 路由路径
 * @param position 滚动位置
 */
export function saveScrollPosition(path: string, position: number): void {
  scrollPositionMap.set(path, position)
}

/**
 * 获取滚动位置
 * @param path 路由路径
 */
export function getScrollPosition(path: string): number | undefined {
  return scrollPositionMap.get(path)
}

/**
 * 清除滚动位置
 * @param path 路由路径（可选，不传则清除所有）
 */
export function clearScrollPosition(path?: string): void {
  if (path) {
    scrollPositionMap.delete(path)
  } else {
    scrollPositionMap.clear()
  }
}

/**
 * 路由滚动行为配置
 */
export const scrollBehavior: RouterScrollBehavior = (to, from, savedPosition) => {
  return new Promise((resolve) => {
    console.log('触发路由滚动')

    // 1. 如果有浏览器保存的位置（前进/后退），优先使用
    if (savedPosition) {
      resolve(savedPosition)
      return
    }

    // 2. 如果有锚点， 约定 默认不滚动 然后在具体的组件里面onActivated里面 或者watch处理滚动逻辑 以及漫游逻辑等
    if (to.hash) {
      // console.log(to.hash, window.scrollY)
      // resolve({
      //   top: window.scrollY,
      // })

      return
    }

    // 3. 检查是否有保存的滚动位置
    const savedScrollY = scrollPositionMap.get(to.fullPath)
    if (savedScrollY !== undefined) {
      setTimeout(() => {
        resolve({
          top: savedScrollY,
          behavior: 'smooth',
        })
      }, 300)
      return
    }

    // 4. 默认滚动到顶部
    resolve({ top: 0 })
  })
}

// 导出默认配置
export default scrollBehavior
