import { ArticleTypeEnum, ReleaseStatusEnum } from '@/constants'
import UploadFile from '@/components/common/UploadFile/index.vue'
import { useResetData } from '@/hooks'

export default defineComponent((_, { expose }) => {
  const [form, resetForm] = useResetData({
    title: '',
    content: '',
    faceUrl: '',
    releaseStatus: ReleaseStatusEnum.PUBLISH,
    type: ArticleTypeEnum.POST,
  })
  const formRef = ref<InstanceType<typeof ElForm>>()
  const rules = {
    title: [{ required: true, message: '请输入帖子标题', trigger: 'blur' }],
    content: [{ required: true, message: '请输入帖子内容', trigger: 'blur' }],
    faceUrl: [{ required: true, message: '请上传贴图', trigger: 'change' }],
    releaseStatus: [{ required: true, message: '请选择发布时间', trigger: 'blur' }],
  }

  const validate = async () => {
    try {
      await formRef.value?.validate()
      return form.value
    } catch (error) {
      console.log(error)
      ElMessage.warning('请检查输入内容')
      return null
    }
  }

  const resetFields = () => {
    formRef.value?.resetFields()
    resetForm()
  }

  expose({
    validate,
    resetFields,
  })
  return () => (
    <div>
      <el-form
        ref={formRef}
        model={form.value}
        label-width="auto"
        label-position="right"
        size="small"
        rules={rules}
      >
        <el-form-item label="标题" prop="title">
          <el-input
            v-model={form.value.title}
            placeholder="请输入帖子标题"
            maxlength={30}
            show-word-limit
          />
        </el-form-item>
        <el-form-item label="内容" prop="content">
          <el-input
            v-model={form.value.content}
            type="textarea"
            placeholder="良言良语，快乐无限；恶语恶言，伤心伤肝。请简述发帖内容，或点击长文章编辑格式"
            rows={6}
            maxlength={255}
            show-word-limit
            class="content-input"
          />
        </el-form-item>
        <el-form-item label="图片" prop="faceUrl">
          {/* @ts-ignore */}
          <UploadFile v-model={form.value.faceUrl} />
        </el-form-item>
        <el-form-item label="发布时间" prop="releaseStatus">
          <el-radio-group v-model={form.value.releaseStatus} class="radio-group">
            <el-radio value={ReleaseStatusEnum.PUBLISH} class="radio-item immediate">
              立即发布
            </el-radio>
            <el-radio value={ReleaseStatusEnum.DRAFT} class="radio-item scheduled">
              定时发布
            </el-radio>
          </el-radio-group>
        </el-form-item>
      </el-form>
    </div>
  )
})
