import { ArticleTypeEnum, ReleaseStatusEnum, SendTypeEnum } from '@/constants'

import UploadFile from '@/components/common/UploadFile/index.vue'
import SelectTags from '@/components/common/SelectTags/index.vue'

import { useResetData } from '@/hooks'
import type { TagItemDto } from '@/api/tag/types'

export default defineComponent((_, { expose }) => {
  const [form, resetForm] = useResetData({
    title: '',
    content: '',
    faceUrl: '',
    releaseStatus: ReleaseStatusEnum.PUBLISH,
    type: ArticleTypeEnum.PRACTICE,
    mainTagId: '',
    tagList: [],
    sendType: SendTypeEnum.IMMEDIATE,
    sendTime: '',
  })
  const formRef = ref<InstanceType<typeof ElForm>>()
  const rules = {
    title: [{ required: true, message: '请输入帖子标题', trigger: 'blur' }],
    content: [{ required: true, message: '请输入帖子内容', trigger: 'blur' }],
    faceUrl: [{ required: true, message: '请上传贴图', trigger: 'change' }],
    releaseStatus: [{ required: true, message: '请选择发布时间', trigger: 'blur' }],
    mainTagId: [{ required: true, message: '请选择主标签', trigger: 'blur' }],
    sendType: [{ required: true, message: '请选择发布类型', trigger: 'blur' }],
    sendTime: [{ required: true, message: '请选择发布时间', trigger: 'blur' }],
  }

  const transformForm = () => {
    return {
      ...form.value,
      tagList: [form.value.mainTagId, ...form.value.tagList].map((tag, index) => {
        return {
          sort: index,
          tagId: Number(tag),
        }
      }),
    }
  }

  const validate = async () => {
    try {
      await formRef.value?.validate()
      console.log(transformForm())
      return transformForm()
    } catch (error) {
      console.log(error)
      ElMessage.warning('请检查输入内容')
      return null
    }
  }

  const filterTagsFn = (allTags: TagItemDto[]) => {
    // 引用了form.value.mainTagId
    return allTags.filter((tag) => tag.id !== Number(form.value.mainTagId))
  }

  const resetFields = () => {
    formRef.value?.resetFields()
    resetForm()
  }

  expose({
    validate,
    resetFields,
  })
  return () => (
    <div>
      <el-form
        ref={formRef}
        model={form.value}
        label-width="auto"
        label-position="right"
        rules={rules}
      >
        <el-form-item label="标题" prop="title">
          <el-input
            v-model={form.value.title}
            placeholder="请输入实践标题"
            maxlength={200}
            show-word-limit
          />
        </el-form-item>
        <el-form-item label="内容" prop="content">
          <el-input
            v-model={form.value.content}
            type="textarea"
            placeholder="分享你的企业文化实践实例"
            rows={6}
            maxlength={1000}
            show-word-limit
            class="content-input"
          />
        </el-form-item>
        <el-form-item label="图片" prop="faceUrl">
          {/* @ts-ignore */}
          <UploadFile v-model={form.value.faceUrl} />
        </el-form-item>
        <el-form-item label="主标签" prop="mainTagId">
          {{
            // @ts-ignore
            default: () => <SelectTags v-model={form.value.mainTagId} />,
            label: () => (
              // <el-tooltip content="主标签最多选1个" placement="top">
              <span class="cursor-pointer">
                副标签
                {/* <el-icon class="ml-1">
                  <InfoFilled />
                </el-icon> */}
              </span>
              // </el-tooltip>
            ),
          }}
        </el-form-item>
        <el-form-item label="副标签">
          {{
            default: () => (
              // @ts-ignore
              <SelectTags
                v-model={form.value.tagList}
                filterTagsFn={filterTagsFn}
                maxSelectedTags={3}
              />
            ),
            label: () => (
              // <el-tooltip content="副标签最多选3个" placement="top">
              <span class="cursor-pointer">
                副标签
                {/* <el-icon class="ml-1">
                    <InfoFilled />
                  </el-icon> */}
              </span>
              // </el-tooltip>
            ),
          }}
        </el-form-item>
        <el-form-item label="发布类型" prop="sendType">
          <el-radio-group v-model={form.value.sendType} class="radio-group">
            <el-radio value={SendTypeEnum.IMMEDIATE} class="radio-item immediate">
              立即发布
            </el-radio>
            <el-radio value={SendTypeEnum.SCHEDULED} class="radio-item scheduled">
              定时发布
            </el-radio>
          </el-radio-group>
        </el-form-item>
        {form.value.sendType === SendTypeEnum.SCHEDULED && (
          <el-form-item label="发布时间" prop="sendTime">
            <el-date-picker
              class="ml-2"
              v-model={form.value.sendTime}
              type="datetime"
              placeholder="请选择发布时间"
            />
          </el-form-item>
        )}
      </el-form>
    </div>
  )
})
